#ifdef SOUND


// Declare Import for Java Sound

   import javax.sound.sampled.*;


/** 
 *
 * Class for the Sound Engine used by SoundInterface Class.
 *
 * @author  David de Niese
 * @version 0.56f
 * @final   TRUE
 *
 */

public final class SoundEngine 
{



     /**
      *
      * <P>The number of samples a second.</P>
      *
      */

      protected static final int soundFreq = 11025;



     /**
      *
      * <P>The number of Refreshs a Second.</P>
      *
      */

      protected static final int soundRefreshRate = 50;



     /**
      *
      * <P>The buffer for the Sound Data.</P>
      *
      */

      private byte buffer[] = new byte[512];



   
     /**
      *
      * <P>The current NES Machine.</P>
      *
      */

      protected NES nes;


     /**
      *
      * <P>The current Graphical User Interface.</P>
      *
      */

      private GUI gui;



     /**
      *
      * <P>The Sampler Line that Transmits data to the Speakers.</P>
      *
      */

      private SourceDataLine line;



     /**
      *
      * <P>The output Stream that is used when recording to WAV Format.</P>
      *
      */

      private java.io.FileOutputStream rawData;



     /**
      *
      * <P>Whether or not the current sound data is being recorded.</P>
      *
      */

      private boolean recordMode = false;


     /**
      *
      * <P>The number of bytes received that need to be written to the WAV File.</P>
      *
      */

      private int bytesReceived = 0;



     /**
      *
      * <P>Create a new Sound Engine.</P>
      *
      */

      public SoundEngine(NES nes, GUI gui) throws Exception 
      {


         // Set Reference to NES and GUI

            this.nes = nes;
            this.gui = gui;



         // Try and Enable Sound
         
            try
            {
               
            
               // Create 11025Hz 8-bit Audio Format
      
                  AudioFormat format = new AudioFormat(SoundEngine.soundFreq,8,1,false,false);
      
      
               // Check if the Format is Supported on this System
      
                  DataLine.Info info = new DataLine.Info(SourceDataLine.class, format);
                  if (!AudioSystem.isLineSupported(info)) throw new Exception();
      
      
               // Capture the Line and prepare it Ready for Data
      
                  line = (SourceDataLine) AudioSystem.getLine(info);
                  line.open(format);
                  line.start();
                  
                  
            }
            catch (Error err)
            {
               throw new Exception("Sound Not Supported");
            }
            catch (java.io.IOException e)
            {
               throw new Exception("Sound Not Supported");
            }


      }



#ifdef STAND


     /**
      *
      * <P>Request that the sound data should be recorded.</P>
      *
      */

      public final void recordStart() 
      {


         try {

            // Check if already Recording

               if (recordMode) {

                  // Calculate the Number of Seconds of already Recorded Sound

                     int secs = (int)((double)bytesReceived / (double)SoundEngine.soundFreq);


                  // Inform the User

                     gui.writeToScreen("Recording " + secs + " seconds...");
                     return;

               }


            // Create the RAW Data File

               rawData = new java.io.FileOutputStream("input.raw");
               bytesReceived = 0;
               recordMode = true;


            // Inform the User that Recording has Started

               gui.writeToScreen("Recording Started...");


         } catch (Exception e) {}


      }




     /**
      *
      * <P>Request that the sound recording should stop.</P>
      *
      */

      public final void recordStop() {


         // Check if Currently Recording

            if (rawData == null) return;


         // Calculate the Length of the Recording

            int secs = (int)((double)bytesReceived / (double)SoundEngine.soundFreq);


         // Save the Recording

            try 
            {


               // Determine the Amount of Data Received

                  int numFrames = bytesReceived;


               // Turn off Record Mode and Close Files

                  recordMode = false;
                  rawData.close();
                  rawData = null;


               // Create a Suitable 11025Hz 8-bit PCM Audio Format

                  AudioFormat format = new AudioFormat(SoundEngine.soundFreq,8,1,false,false);


               // Create the Wav File for Output

                  java.io.File fileOut = new java.io.File("output.wav");
                  java.io.FileOutputStream outputStream = new java.io.FileOutputStream(fileOut);


               // Open the RAW Data File for Input

                  java.io.InputStream inputStream = new java.io.FileInputStream("input.raw");


               // Set the Format for Conversion

                  AudioFileFormat.Type fileType = AudioFileFormat.Type.WAVE;
                  AudioInputStream audioInputStream = new AudioInputStream(inputStream,format,numFrames);


               // Write and Convert the InputStream to WAV Format

                  AudioSystem.write(audioInputStream,fileType,fileOut);


               // Close the InputStream to the WAV File

                  audioInputStream.close();


               // Delete the RAW Data File

                  java.io.File oldFile = new java.io.File("input.raw");
                  oldFile.delete();


               // Inform the User of the Successful Recording

                  gui.writeToScreen("Recording of " + secs + " seconds saved as 'output.wav'");


            } 
            catch (Exception e) 
            {

               // Inform the User of the Unsuccessful Recording

                  gui.writeToScreen("Recording of " + secs + " seconds failed.");


            }

      }





#endif



     /**
      *
      * Current Position in Buffer
      *
      */
      
      int bufferpointer = 0;


     /**
      *
      * The Number of Bytes Written to the Line for Tracking how often to Drain It
      *
      */
      
      int bytesWrittenToLine = 0;



     /**
      *
      * <P>Write an 16-bit signed word of Sound Data to the Line.</P>
      *
      */

      public final void write(int value) 
      {


         // Convert to 8-bit Unsigned

            int output = (value >> 8) ^ 0x80;
            output &= 0xFF;


         // Check the Line is Available

            if (line == null) return;
            


         // Record Byte in Buffer

            buffer[bufferpointer++] = (byte)output;
            
#ifdef STAND

            if (recordMode) recordByte(output);

#endif



         // Only Proceed if Buffer not Full
                     
            if (bufferpointer<buffer.length)
                return;
         
         
         
         // Buffer Full so Attempt Play
                
            bufferpointer=0;

            try 
            { 

               
               // Write Byte to Line
               
                  line.write(buffer,0,buffer.length);
               
               
               // Drain line Every 5 Seconds
               
                  if (++bytesWrittenToLine == (soundFreq * 5))
                  {
                     
                     line.drain();
                     bytesWrittenToLine = 0;
                     
                  }


            } catch (Exception e) {}


      }


#ifdef STAND

     /**
      *
      * <P>Record a 8-bit Unsigned byte of Sound Data to a RAW File.</P>
      *
      */

      private final void recordByte(int output) 
      {


         try 
         {


            // Write the Byte to INPUT.RAW

               if (rawData != null) rawData.write(output);


            // Increment the Bytes

               bytesReceived++;


         } catch (Exception e) {

              recordStop();

         }

      }

#endif

      


}

#endif
