#ifdef STAND


// Declare Imports

   import java.awt.*;
   import java.awt.image.PixelGrabber;
   import java.io.*;




/** 
 *
 * Class for Saving Images in TGA Format used by the NESCafe NES Emulator.
 *
 * @author   David de Niese
 * @version  0.56f
 * @final    TRUE
 *
 */

public final class SaveImage extends Canvas 
{


     /**
      *
      * <P>File Format extension.</P>
      *
      */

      protected static String extension = "TGA";


     /**
      *
      * Use for Static Reference
      *
      */
      
      public SaveImage() throws Exception 
      {
      }
      
      
      
     /**
      *
      * <P>Create a new SaveImage object.</P>
      *
      * @param Image The image to save.
      * @param outputstream The output stream to save it to.
      *
      */

      public SaveImage(Image image) throws Exception 
      {


         // Declare SaveAs File and FileName

            File saveAs;
            String fileName;


         // Find Available Image Number

            int saveNum = 0;

            do {

               // Build FileName

                  fileName = "" + saveNum;
                  while (fileName.length() < 4) fileName = "0" + fileName;
                  fileName = "NESCafe - " + fileName + "." + SaveImage.extension.toLowerCase();


               // Create File

                  saveAs = new File(fileName);


               // Increment Counter incase it exists

                  saveNum++;


            } while (saveAs.exists());



         // Make TGA Data Block
         
            byte[] datablock = SaveImage.makeTGA(image);
            
            
         // Grab reference to Output Stream

            OutputStream output = new FileOutputStream(saveAs.toString());
            output.write(datablock);
            output.close();
              

      }



     /**
      *
      * <P>Make TGA File</P>
      *
      */
      
      public static final byte[] makeTGA24(Image image) throws Exception
      {
         
         
         // Create Non Static Reference
         
            SaveImage ref = new SaveImage();
         

         // Determine the Width and Height of the Image
       
            int imgWidth = (short)image.getWidth(ref);
            int imgHeight = (short)image.getHeight(ref);


         // Destroy Non Static Reference

            ref = null;
            

         // Grab the Pixels

            int size = imgWidth * imgHeight;
            int data[] = new int[size];
            PixelGrabber pixelGrabber = new PixelGrabber(image,0,0,imgWidth,imgHeight, data, 0, imgWidth);
            pixelGrabber.grabPixels();


         // Create Out Buffer
         
            byte[] outputBuffer = new byte[18 + 3*size];
            
            
         // Write TGA Header

            outputBuffer[0] = 0;  // Image Field Length
            outputBuffer[1] = 0;  // No Colour Map
            outputBuffer[2] = 2;  // Image Type
            outputBuffer[3] = 0;  // Future Expansion (5 bytes)
            outputBuffer[4] = 0;
            outputBuffer[5] = 0;
            outputBuffer[6] = 0;
            outputBuffer[7] = 0;


         // Write the X and Y Origin of the Image
          
            outputBuffer[ 8] = 0;  // X origin
            outputBuffer[ 9] = 0;
            outputBuffer[10] = 0;  // Y origin
            outputBuffer[11] = 0;



         // Write the Width and Height

            outputBuffer[12] = (byte)( imgWidth & 0xFF );
            outputBuffer[13] = (byte)( (imgWidth >> 8) & 0xFF );

            outputBuffer[14] = (byte)( imgHeight & 0xFF );
            outputBuffer[15] = (byte)( (imgHeight >> 8) & 0xFF );


         // Write Attribute Information

            outputBuffer[16] = 24;  // 24 bit colour
            outputBuffer[17] =  0;  // Attribute Data



         // Go through the Image Data

            byte blue = 0;
            byte green = 0;
            byte red = 0;
            
            int spixel = 0;
            int dpixel = 0;  //after flip
            

         // Start at the Bottom and Encode Up

            for (int y = imgHeight-1; y >= 0; y--) 
            {

               for (int x=0; x<imgWidth; x++ ) 
               {


                  // Find the Pixel Offset

                     spixel = y * imgWidth + x;
                     dpixel = ((imgHeight-1-y) * imgWidth + x)*3+18;


                  // Determine the RGB Components

                     red   = (byte)((data[spixel] >> 16) & 0xff);
                     green = (byte)((data[spixel] >>  8) & 0xff);
                     blue  = (byte)((data[spixel]      ) & 0xff);


                  // Write the RGB Components in Reverse Order

                     outputBuffer[dpixel + 0] = blue;
                     outputBuffer[dpixel + 1] = green;
                     outputBuffer[dpixel + 2] = red;


               }


            }
            
           
            return outputBuffer;
            
      }



#ifdef FUTURE

     /**
      *
      * <P>Make BMP File</P>
      *
      */
      
      public static final byte[] makeBMP(Image image) throws Exception
      {
         return null;
         
         
      }
      
#endif

      
     /**
      *
      * <P>Make TGA File 8-bits Per Pixel Plus Palette</P>
      *
      */
      
      public static final byte[] makeTGA(Image image) throws Exception
      {
         
         // Palette Size
         
            int palSize = 256;
            
            
         // Create Non Static Reference
         
            SaveImage ref = new SaveImage();
         

         // Determine the Width and Height of the Image
       
            int imgWidth = (short)image.getWidth(ref);
            int imgHeight = (short)image.getHeight(ref);


         // Destroy Non Static Reference

            ref = null;
            

         // Grab the Pixels

            int size = imgWidth * imgHeight;
            int data[] = new int[size];
            PixelGrabber pixelGrabber = new PixelGrabber(image,0,0,imgWidth,imgHeight, data, 0, imgWidth);
            pixelGrabber.grabPixels();


         // Create Out Buffer
         
            byte[] outputBuffer = new byte[18 + palSize*3 + size];
            
            
         // Write TGA Header

            outputBuffer[0] = 0;                             // Image Field Length
            outputBuffer[1] = 1;                             // Has Palette
            outputBuffer[2] = 1;                             // Palette Indexed Image
            outputBuffer[3] = 0;                             // First Map Entry in Palette
            outputBuffer[4] = 0;  
            outputBuffer[5] = (byte)(palSize & 0xFF);        // Last Map Entry in Palette
            outputBuffer[6] = (byte)((palSize>>8) & 0xFF);
            outputBuffer[7] = 24;                            // Bits Per Pixel for Palette Entries


         // Write the X and Y Origin of the Image
          
            outputBuffer[ 8] = 0;  // X origin
            outputBuffer[ 9] = 0;
            outputBuffer[10] = 0;  // Y origin
            outputBuffer[11] = 0;



         // Write the Width and Height

            outputBuffer[12] = (byte)( imgWidth & 0xFF );
            outputBuffer[13] = (byte)( (imgWidth >> 8) & 0xFF );

            outputBuffer[14] = (byte)( imgHeight & 0xFF );
            outputBuffer[15] = (byte)( (imgHeight >> 8) & 0xFF );


         // Write Attribute Information

            outputBuffer[16] =  8;  // 8 bits per pixel
            outputBuffer[17] =  0;  // Attribute Data

            
         
            byte[] palette = new byte[3*palSize];
            
            

         // Go through the Image Data

            byte blue = 0;
            byte green = 0;
            byte red = 0;
            
            int spixel = 0;
            int dpixel = 0;  //after flip
            

            byte lastPaletteEntry = 0;


         // Start at the Bottom and Encode Up

            for (int y = imgHeight-1; y >= 0; y--) 
            {

               for (int x=0; x<imgWidth; x++ ) 
               {


                  // Find the Pixel Offset

                     spixel = y * imgWidth + x;
                     dpixel = ((imgHeight-1-y) * imgWidth + x)+(palSize*3)+18;


                  // Determine the RGB Components

                     red   = (byte)((data[spixel] >> 16) & 0xff);
                     green = (byte)((data[spixel] >>  8) & 0xff);
                     blue  = (byte)((data[spixel]      ) & 0xff);


                  // Find Entry in Palette
                  
                     boolean found = false;
                     byte palEntry = 0;
                     
                     for (int i=0; i<lastPaletteEntry; i++)
                     {
                        if (blue==palette[i*3 + 0] && green==palette[i*3 + 1] && red==palette[i*3 + 2])
                        {
                           found = true;
                           palEntry = (byte)i;
                           break;
                        }
                        
                     }
                     
                     
                     
                  // Use Existing Palette Entry or Create new One
                  
                     if (found)
                     {
                        outputBuffer[dpixel] = palEntry;
                     }
                     else
                     {
                        outputBuffer[dpixel] = lastPaletteEntry;
                        palette[lastPaletteEntry*3+0] = blue;
                        palette[lastPaletteEntry*3+1] = green;
                        palette[lastPaletteEntry*3+2] = red;
                        
                        if (lastPaletteEntry<(palSize-1))
                        {                       
                           lastPaletteEntry++;
                        }
                        
                     }
                    

               }


            }
            
            
         // Write the Palette
         
            for (int i=0; i<palette.length; i++)
            {
               outputBuffer[18+i] = palette[i];
            }
           
            return outputBuffer;
            
      }




}

#endif
