
/** 
 *
 * Class for the NES Machine used by NESCafe.
 *
 * @author   David de Niese
 * @version  0.56f
 * @final    TRUE
 *
 */

public final class NES 
{



     /**
      *
      * <P>The User Settings.</P>
      *
      */

      protected UserSettings usersettings = new UserSettings();


     /**
      *
      * <P>Controller 1 Joypad.</P>
      *
      */

      protected JoyPad joyPad1;


     /**
      *
      * <P>Controller 2 Joypad.</P>
      *
      */

      protected JoyPad joyPad2;




#ifndef MOBILE


     /**
      *
      * <P>The Central Processing Unit (CPU).</P>
      *
      */

      protected CPU cpu;





#ifdef APPLET


     /**
      *
      * <P>The NESCafe Applet When in Applet Mode</P>
      *
      */

      protected NESCafeApplet applet;


#endif


     /**
      *
      * <P>The Graphical User Interface (GUI).</P>
      *
      */

      protected GUI gui;


     /**
      *
      * <P>The Memory Mapper.</P>
      *
      */

      protected Mapper mapper;


     /**
      *
      * <P>The Memory Manager.</P>
      *
      */

      protected MemoryManager memory;


     /**
      *
      * <P>The Picture Processing Unit (PPU).</P>
      *
      */

      protected PPU ppu;


#ifdef SOUND

     /**
      *
      * <P>The Sound Processor.</P>
      *
      */

      protected SoundInterface sound = null;

#endif


     /**
      *
      * <P>The current Palette.</P>
      *
      */

      protected Palette palette = new Palette();


     /**
      *
      * <P>Whether or not a Cartridge is currently loaded.</P>
      *
      */

      private boolean noCartLoaded = true;


     /**
      *
      * <P>The Filename of the current Cartridge.</P>
      *
      */

      protected String currentCartFileName="";


     /**
      *
      * <P>The Filename of the last Opened Cartridge.</P>
      *
      */

      public String lastOpenedCartFileName = ".";


     /**
      *
      * Last Recorded Error Message
      *
      */

      protected String errorMessage = "";


     /**
      *
      * Frame IRQ
      *
      */
      
      public int frameIRQEnabled = 0xFF;
      

     /**
      *
      * Frame IRQ Disenable
      *
      */
      
      public int frameIRQDisenabled = 0;



#ifdef DEBUG

     /**
      *
      * Register Debugger
      *
      */
      
      public boolean regdebug = false;


#endif

      
     /**
      *
      * <P>Create a new Nintendo Machine.</P>
      *
      */

      public NES() 
      {

         noCartLoaded = true;
         currentCartFileName = "";

      }


#ifdef APPLET

     /**
      *
      * <P>Set Applet Mode</P>
      *
      */

      public final void setAppletMode(NESCafeApplet nescafeApplet)
      {

         this.applet = nescafeApplet;

      }


#endif


     /**
      *
      * <P>Initialise the Nintendo Machine.</P>
      *
      * @param theGUI The Graphical User Interface in use.
      *
      */

      public final void init(GUI gui) 
      {


         // Grab Reference to the Current Graphical User Interface

            this.gui = gui;


         // Create Controller 1

            joyPad1 = new JoyPad(JoyPad.JOYPAD_1);


         // Create Controller 2

            joyPad2 = new JoyPad(JoyPad.JOYPAD_2);


         // Create a CPU

            cpu = new CPU(this,gui);


         // Create a PPU

            ppu = new PPU(this,gui); 


         // Create a Memory Manager

            memory = new MemoryManager(this,gui); 


         // Not all JVMs Support this

#ifdef SOUND

            try 
            {

               // Create a Sound Processor

                 sound = new SoundInterface(this,gui);

            } catch (Exception exception) {} catch (Error error) {}

#endif


         // Not all JVMs Support this

            usersettings.loadSettings();

      }


     /**
      *
      * <P>Load a cartridge and start it running.</P>
      *
      * @param fileName The filename of the cartridge to load.
      *
      * @return True if the Cartridge was successfully loaded.
      *
      */

      public final boolean cartLoad(java.io.File fileName) 
      {

	     if (fileName == null || fileName.equals(""))
	     	return cartLoadDefault();
	     	
         return cartLoad(fileName.toString());

      }


     /**
      *
      * <P>Load default cartridge and start it running.</P>
      *
      */

	  public final boolean cartLoadDefault()
	  {
	 
	     return cartLoad("");
	  }


     /**
      *
      * <P>Load a cartridge and start it running.</P>
      *
      * @param fileName The filename of the cartridge to load.
      *
      * @return True if the Cartridge was successfully loaded.
      *
      */

      public final boolean cartLoad(String fileName) {


         // Clear the Screen

            gui.tvController.deleteDisplay();


            try 
            {

               // Load the ROM from the File
  
                  cpu.cpuLoadRom(fileName);


            } catch (NESCafeException e) {


               // Record the Error Message

                  errorMessage = e.getMessage();


               // Show the Error Message and Return

#ifdef STAND

                  DavOS.MsgBox(errorMessage);


#endif

                  return false;

            }



         // Stop the Currently Loaded ROM

            cpu.cpuStop();
            noCartLoaded = true;
            
#ifdef STAND

            gui.showMenuNoRom();

#endif


         // Run the new ROM

            cpu.cpuRun();
            noCartLoaded = false;
            
#ifdef STAND

            gui.showMenuRom();

#endif


         // Return Signal

            currentCartFileName = fileName;
            return true;


      }



     /**
      *
      * <P>Unload the currently running cartridge.</P>
      *
      */

      public final void cartUnload() 
      {


         // Check if a Cart is Loaded

            if (noCartLoaded) return;


         // Stop the Current CPU Processing

            cpu.cpuStop();



#ifdef STAND


         // Save SaveRAM

            memory.saveSaveRAM();


         // Show No ROM Menu
   
            gui.showMenuNoRom();



#ifdef SOUND


         // If any Sound Recording was Happening then Stop it

            if (gui.soundMenu2.getLabel().equals("Stop Recording WAV")) 
            {

               gui.soundMenu2.setLabel("Start Recording WAV");
               sound.recordStop();

            }

#endif


         // Turn Off Debug

            cpu.debugExit();

#endif


         // Clean Up the Resources

            noCartLoaded = true;
            currentCartFileName = "";
            System.gc();

      }


     /**
      *
      * <P>Determine whether or not a cartridge is currently running.</P>
      *
      * @return True if a Cartridge is currently running, else False.
      *
      */

      public final boolean isCartRunning() 
      {

         return !noCartLoaded;

      }


     /**
      *
      * <P>Determine the name of the currently loaded cartridge.</P>
      *
      * @return The filename of the currently loaded cartridge.
      *
      */

      public final String getCartFileName() 
      {

         return currentCartFileName;

      }


     /**
      *
      * <P>Reset the current Processor.</P>
      *
      */

      public final void reset() 
      {

         cpu.cpuReset();

      }




     /**
      * 
      * <P>Loads state of the NES from the last committed state for the Cart.</P>
      *
      * This method is called by the Processor to avoid corruption.
      *
      */

      public final void stateLoad() 
      {



         // Turn off the Request Submitted by the User

            gui.loadStateRequest = false;



         // Declare FileInputStream Object

            java.io.InputStream input;



         // Determine the Filename of the State File
         
            
            try 
            {

#ifdef APPLET


               // Determine Filename
               
                  String fileName = applet.loadStateFilename;
                  
                  
               // Attempt to Bind to URL
               
                  java.net.URL url = new java.net.URL(applet.getCodeBase() + fileName);
                  input = new java.util.zip.GZIPInputStream(new java.io.BufferedInputStream(url.openStream()));


#endif

#ifdef STAND
               
               // Determine the Original filename of the Cartridge (without extension)
      
                  String fileName = DavOS.getFileName(currentCartFileName);


               // Attempt to Open Save Directory

                  java.io.File saveDirectory = new java.io.File("savestates");


               // Attempt to Open an InputStream from the File

                  java.io.File inputFile = new java.io.File(saveDirectory + "/NESCafe - " + fileName + ".nss");
                  input = new java.util.zip.GZIPInputStream(new java.io.FileInputStream (inputFile));

               
#endif

            } 
            catch (java.io.FileNotFoundException e) 
            {

#ifdef STAND

               // Inform User that no Committed State was Found

                  DavOS.MsgBox("You have not committed a State File for this game.");

                  
#endif
                  return;


            } 
            catch (Exception e) 
            {


#ifdef STAND

               // Inform User of Difficulties when Opening File

                  DavOS.MsgBox("A Committed State could not be loaded...");

                  
#endif  
                  return;

            }


            try 
            {


               // Check the Signature
               
                  byte[] signature = new byte[7];
                  
                  if (input.read(signature,0,7) != 7 || !(new String(signature)).equals("NESCafe"))
                  {
                     throw new NESCafeException("Invalid Save-State File Format.");
                  }
               
               
               
               // Check Version Number
               
                  if (input.read() > 1)
                  {
                     throw new NESCafeException("File Format is newer than supported by this version of NESCafe.");
                  }



               // Load the CPU State

                  cpu.stateLoad(input);


               // Load the Memory Manager State

                  memory.stateLoad(input);


               // Load the PPU State 

                  ppu.stateLoad(input);


               // Check that the Mappers Match

                  int readMapNum = input.read();


               // Load the Mapper State

                  mapper.stateLoad(input);


               // Load the JoyPad State

                  joyPad1.stateLoad(input);


#ifdef SOUND

               // Reset the Sound Hardware

                  if (sound !=null) sound.reset();

#endif

               // Close the InputStream

                  input.close();


            } 
            catch (NESCafeException e)
            {

#ifdef STAND

               DavOS.MsgBox("The NESCafe Saved-State is invalid: " + e.getMessage());

#endif
               return;
               
            }
            catch (java.io.IOException e) 
            {

#ifdef STAND

               DavOS.MsgBox("The NESCafe Saved-State file could not be read.");

#endif
               return;

            }


#ifdef STAND

         // Un Pause the CPU (unless in Applet Mode)

            cpu.setCPUPause(false);


#endif

#ifndef PUNCHOUT

         // Inform the User that the State Loaded Correctly

            gui.writeToScreen("State Successfully Loaded...");


#endif

      }


     /**
      * 
      * <P>Saves state of the NES to disk.</P>
      *
      * This method is called by the Processor to avoid corruption.
      *
      */

      public final void stateSave() 
      {


#ifdef STAND


         // Turn off User Request

            gui.saveStateRequest = false;


         // Determine the Original filename of the Cartridge (without extension)

            String fileName = DavOS.getFileName(currentCartFileName);


         // Declare FileInputStream Object

            java.io.OutputStream output;


            try 
            {


               // Attempt to Create Save Directory

                  java.io.File saveDirectory = new java.io.File("savestates");

                  if (saveDirectory.exists() && !saveDirectory.isDirectory())
                  {
                     saveDirectory.delete();
                  }

                  if (!saveDirectory.exists())
                  {
                     saveDirectory.mkdir();

                  }


               // Attempt to Open an OutputStream to the File

                  java.io.File outputFile = new java.io.File(saveDirectory + "/NESCafe - " + fileName + ".nss");
                  output = new java.util.zip.GZIPOutputStream(new java.io.FileOutputStream (outputFile));


            } 
            catch (Exception e) 
            {


               // Inform User of Difficulties when Saving File

                  DavOS.MsgBox("A Committed State could not be saved.");
                  return;


            }

            try {


               // Write 7-byte Signature
               
                  output.write("NESCafe".getBytes());
                  
                  
               // Write Save State Version
               
                  output.write(1);
                  
                  
               // Save CPU State

                  cpu.stateSave(output);


               // Save Memory State

                  memory.stateSave(output);


               // Save PPU State

                  ppu.stateSave(output);


               // Write Mapper Number

                  output.write(mapper.getMapperNumber() & 0xFF);


               // Save Mapper State

                  mapper.stateSave(output);
                  

               // Save JoyPad State

                  joyPad1.stateSave(output);
                  joyPad2.stateSave(output);


               // Close the File

                  output.close();



            } 
            catch (java.io.IOException e) 
            {

               DavOS.MsgBox("The NESCafe Saved-State File could not be Saved.");
               return;

            }


         // Inform User of Progress

            gui.writeToScreen("State Successfully Saved...");


#endif

      }


#endif

}

