#ifndef MOBILE


// Declare Imports

   import java.io.*;
   import java.net.*;
   import java.awt.*;


/** 
 *
 * Class for the Server Connection used by the NESCafe NES Emulator.
 *
 * @author   David de Niese
 * @version  0.56f
 * @final    TRUE
 *
 */

public final class ServerConnection 
{



#ifdef STAND

//////////////////////////////////////////////////////////////////////
//
// Server Location Information
//
//////////////////////////////////////////////////////////////////////


     /**
      * 
      * <P>The NES ROM Server.</P>
      *
      */

      public static final String NESRomServer = "www.davieboy.net";



     /**
      *
      * <P>The Directory on the ROM Server that contains the ROMs.</P>
      *
      */

      public static final String NESCafeRoot = "/nescafe/";



     /**
      *
      * <P>The Directory on the ROM Server that contains the ROMs.</P>
      *
      */

      public static final String NESRomDirectory = "/nescafe/server/";


     /**
      * 
      * <P>The filename of the Available Rom List.</P>
      *
      */

      public static final String NESRomList = "NESCafe.romlist";



//////////////////////////////////////////////////////////////////////
//
// Local Connection Information
//
//////////////////////////////////////////////////////////////////////


     /**
      *
      * <P>Whether a Server Connection has been established.</P>
      *
      */

      protected boolean connected = false; 


     /**
      *
      * <P>Local computer name.</P>
      *
      */

      protected String myName="";



     /**
      *
      * <P>The list of all users in the NESCafe Server.</P>
      *
      */

      protected String userList="";



//////////////////////////////////////////////////////////////////////
//
// The Server Side Connections
// 
//////////////////////////////////////////////////////////////////////


     /**
      *
      * <P>The Output Print Stream to the Server.</P>
      *
      */

      private PrintStream out; 


     /**
      *
      * <P>The Input Print Stream from the Server.</P>
      *
      */

      private BufferedReader in  ; 



//////////////////////////////////////////////////////////////////////
//
// The Local Side Connections
//
//////////////////////////////////////////////////////////////////////


     /**
      *
      * <P>The current NES Engine.</P>
      *
      */

      private NES nes; 


     /**
      *
      * <P>The current Graphical User Interface.</P>
      *
      */

      private GUI gui; 


     /**
      * 
      * <P>The Thread that reads Messages from the Server.</P>
      *
      */

      private Thread reader; 


//////////////////////////////////////////////////////////////////////
//
// Windows that are Controller by the Server Connection
//
//////////////////////////////////////////////////////////////////////


     /**
      *
      * <P>The Chat Window.</P>
      *
      */

      public ChatWindow chatWindow; 





//////////////////////////////////////////////////////////////////////
//
// Class Functions
//
//////////////////////////////////////////////////////////////////////


     /**
      *
      * <P>Create a new Server Connection.</P>
      *
      * @param Nes The current NES Engine. 
      * @param Gui The current Graphical User Interface. 
      *
      */

      public ServerConnection (NES nes, GUI gui) 
      {



         // Set the References

            this.nes = nes; 
            this.gui = gui; 



            try {


               // Attempt to Make a Socket Connection on port 7529

                  Socket socket = new Socket(nes.usersettings.coffeePotServer, 7529); 


               // Create Output Stream to Server
       
                  out = new PrintStream(socket.getOutputStream());


               // Grab Input Stream from Server

                  in  = new BufferedReader(new InputStreamReader(socket.getInputStream()));


               // Ask if it is the NESCafe Server

                  out.println("NESCAFE-CHECK");


               // Ignore the Welcome Message

                  String response = in.readLine();


               // Receive Response from Server

                  response = in.readLine();


               // Check that it Responded Correctly

                  if (!response.equals("NESCAFE-OK")) 
                     throw new Exception();


               // Read the Hello Line

                  response = in.readLine().toUpperCase();


               // Check that the Format of the HELLO is Correct

                  if (!response.substring(0,6).equals("HELLO "))
                     throw new Exception();


               // Find out what Name the Server Called me

                  myName = response.substring(6); 


               // Create and Start a new Server Reader

                  reader = new ServerReader(in, nes, gui); 
                  reader.setPriority(3); 
                  reader.start();


               // State that we are Connected to the Server

                  connected = true; 


               // Enable Menus for Chat Server Access

                  gui.serverMenu0.setEnabled(true); 


            } catch (UnknownHostException e) {
   
            } catch (ConnectException e) {
   
            } catch (IOException e) {
   
            } catch (Exception e) {

            }



         // State how we are Connected

            if (connected) {

               System.out.println("CoffeePot Server Located [" + myName + "]");

            }

      }




     /**
      *
      * <P>Send Message to Server.</P>
      *
      * @param message The message to be sent to the Server. 
      *
      */

      public final void close() 
      {

         try 
         {

            out.close();
            connected = false;

         } catch (Exception e) {}

      }


     /**
      *
      * <P>Send Message to Server.</P>
      *
      * @param message The message to be sent to the Server. 
      *
      */

      public final void sendMessage(String message) 
      {


         // The message must be structured as hostName:contenttype:data

            try {


               // Check that the Client is Connected

                  if (!connected) return; 
    

               // Send the Length of the Packet to the Server

                  out.print ( (char) ((message.length() >> 0) & 0xFF) ); 
                  out.print ( (char) ((message.length() >> 8) & 0xFF) ); 


               // Send the Actual Message

                  out.print (message); 


            } catch (Exception e) {


               // An error occurred so cancel the Connection

                  connected = false; 


            }

      }



     /**
      *
      * <P>Show the Chat Window.</P>
      *
      */

      public final void showChatWindow() 
      {


         // Check that we have a Connection to the Server

            if (!connected) {

               DavOS.MsgBox("There is currently no connection to the NESCafe Server."); 
               return; 

            }


         // Create a new Chat Window

            if (chatWindow == null) {

               chatWindow = new ChatWindow(nes,gui); 
  
            }


         // Display the Chat Window

            chatWindow.setState(Frame.NORMAL); 
            chatWindow.show();

      }


     /**
      * 
      * <P>Filter the incoming Server Data.</P>
      *
      * @param serverData The incoming Server data. 
      *
      */

      public final void filterNetworkData(String serverData) 
      {


         // Declare Variables for Parsing Data

            String data = serverData; 

            String hostAddress = "";
            String hostName = "";
            String fromAddress = "";
            String contentType = "";
            String message = "";

            String guiMessage = "";


         // Read Host Address

            if (data.indexOf(":") < 0) return; 
            hostAddress = data.substring(0,data.indexOf(":"));
            data = data.substring(data.indexOf(":") + 1); 


         // Determine the Host Name

            if (hostAddress.indexOf(".") >= 0) {

               hostName = hostAddress.substring(0,hostAddress.indexOf("."));

            } else {

               hostName = hostAddress; 

            }



         // Extract the From Address

            if (data.indexOf(":") < 0) return; 
            fromAddress = data.substring(0,data.indexOf(":"));
            data = data.substring(data.indexOf(":") + 1); 


         // Extract the Content Type

            if (data.indexOf(":") < 0) return; 
            contentType = data.substring(0,data.indexOf(":")).toUpperCase();
            message = data.substring(data.indexOf(":") + 1); 


         // Decode the Message



         // MESSAGE : Chat Message

            if (contentType.equals("CHAT") & !message.equals("")) {

               // A Chat Message has been Received

                  if (chatWindow != null) 
                     chatWindow.append (hostName + ": " + message); 

                  return;  

            }


         // MESSAGE : Flush the Input and Output Streams

            if (contentType.equals("FLUSH")) {

               // Flushs the Stream

                  return; 

            }


         // MESSAGE : A User List has been Delivered to Us

            if (contentType.equals("USERLIST")) {

               userList = message; 
               return; 

            }

      }
      
      
      
#endif


#ifdef PUNCHOUT


     /**
      *
      * Punch Out URL
      *
      */
      
      public static final String PUNCHOUT_URL = "http://localhost/play/punchout.submit.asp";
      
      

     /**
      * 
      * <P>Post Data to DavieBoy</P>
      *
      */

      public static boolean postData(String data)
      {

         try
         {
       


            // Send Data
            
               URL url = new URL(ServerConnection.PUNCHOUT_URL);
               URLConnection conn = url.openConnection();
               conn.setDoOutput(true);
               OutputStreamWriter wr = new OutputStreamWriter(conn.getOutputStream());
               wr.write(data);
               wr.flush();
               
               
            // Get Response
            
               BufferedReader rd = new BufferedReader(new InputStreamReader(conn.getInputStream()));
               String line;
               
               boolean foundSuccess = false;
               
               while ((line = rd.readLine()) != null) 
               {
            
                  if (line.equals("SUCCESS"))
                     foundSuccess = true;

               }
               
               wr.close();
               rd.close();


               return foundSuccess;
               

           } 
           catch (Exception e) 
           {}
           catch (Error f)
           {}
               
           return false;
                
      }


#endif


}


#ifdef STAND

/** 
 *
 * Class for the Server Reader used by the Server Connection. 
 *
 * @author   David de Niese
 * @version  0.20
 * @final    TRUE
 *
 */


class ServerReader extends Thread 
{


     /**
      * 
      * <P>The Input Stream from the Server.</P>
      *
      */

      private BufferedReader in; 


     /**
      *
      * <P>The current NES Engine.</P>
      *
      */

      private NES nes; 


     /**
      *
      * <P>The current Graphical User Interface.</P>
      *
      */

      private GUI gui; 



     /**
      *
      * <P>Create a new Server Reader Object.</P>
      *
      * @param Nes The current NES Engine. 
      * @param Gui The current Graphical User Interface. 
      *
      */

      public ServerReader(BufferedReader reader, NES Nes,GUI Gui) {

         nes = Nes; 
         gui = Gui; 

         in = reader; 

      }



      /**
       * 
       * <P>The main Thread for the Server Reader.</P>
       *
       */

       public void run() {


          // Declare Variable for incoming Line

             String line; 


          // Catch Exceptions

             try {


                // Wait for the Server Connection to come Available

                   while (gui.serverConnection == null) {

                      try { sleep(1000); } catch (Exception e) {}

                   }


                // Loop around Receiving and Delivering Content

                   while(true) {


                      // Read the Length of the Incoming Message

                         int contentLength = in.read();
                         contentLength += (in.read() * 256); 


                      // Ignore Bad Content

                         if (contentLength < 0) break; 


                      // Declare Array for Content

                         char[] buffer = new char[contentLength]; 


                      // Read in the Content

                         for (int i = 0; i<contentLength; i++) 
                              buffer[i] = (char)(in.read());
		

                      // Create a String from the Data

                         line = new String(buffer); 


                      // Send String to be Filtered and Delivered

                         gui.serverConnection.filterNetworkData(line); 

                   }

             }

             catch (IOException e) {}



          // Cancel the Connection if an Error Occurred

             gui.serverConnection.connected = false; 


      }

}


#endif

#endif
