#ifndef MOBILE

// Declare Imports

   import java.io.*;
   import java.net.*;


/** 
 *
 * Class for the Memory Manager required by the NESCafe NES Emulator.
 *
 * @author   David de Niese
 * @version  0.56f
 * @final    TRUE
 *
 */

public final class MemoryManager 
{


     /**
      *
      * <P>The current NES Machine.</P>
      *
      */

      protected NES nes;


     /**
      *
      * <P>The current Picture Processing Unit.</P>
      *
      */

      protected PPU ppu;



     /**
      *
      * <P>The current Graphical User Interface.</P>
      *
      */

      private GUI gui;


     /**
      *
      * <P>The main memory.</P>
      *
      */

      protected int memory[] = new int[0x0800];


     /**
      *
      * <P>The Program ROM.</P>
      *
      */

      protected int programROM[] = null;



#ifdef DECOMPILER


     /**
      *
      * Decompiler Executable Code Tags
      *
      */
      
      protected int programROMExecuteTags[] = null;
      
      
      public synchronized void setExecuteCodeMonitor(boolean val)
      {
         executableCodeMonitorEnabled = val;
         
      }

      public synchronized boolean getExecuteCodeMonitor()
      {
         return executableCodeMonitorEnabled;
         
      }
      
      private boolean executableCodeMonitorEnabled = false;
      
      public void setAsExecutableCode(int PC)
      {
         
         if (!getExecuteCodeMonitor())
            return;
            
         
         // Get Offset

            int c = bank[ ((PC & 0xF000) >> 12) ];


         // Set Memory as Executable

            programROMExecuteTags[ c + (PC & 0x0FFF) ] = PC;

      }
      
      public void clearExecutableCode()
      {
         
         for (int i=0; i<programROMExecuteTags.length; i++)
            programROMExecuteTags[i] = -1;
         
      }
      
      public void dumpExecutableCode()
      {
         
         try
         {
            
            
            
            FileOutputStream os = new FileOutputStream (new File("decompiled.txt"));
               
               
               
            Decompiler bob = new Decompiler(programROM);
            
            
            String line = "";
            
            for (int i=0; i<programROM.length; i++)
            {
               if (programROMExecuteTags[i]!=-1)
               {
                  
                  line = DavOS.hex(i,8) + " " + DavOS.hex(programROMExecuteTags[i],4) + " " + bob.decompile(i) + "\r\n";
                  os.write(line.getBytes());
                  
               }
            }
            
            
            os.close();
            
            percentageExecutableCode();
            
            
         }
         catch (Exception e)
         {
            
         }

         
      }

      public void percentageExecutableCode()
      {

         double counter = 0;
         double total   = programROM.length;
                  
         for (int i=0; i<programROM.length; i++)
         {
            if (programROMExecuteTags[i]!=-1)
            {
               counter++;
            }
         }

         double percent = 100*counter/total;
         
         System.out.println("Percentage Analysed: " + (int)percent) ;
         
      }

      
#endif


     /**
      *
      * <P>Save RAM.</P>
      *
      */

      public int saveRAM[] = new int[0x10000];
      
      
      
     /**
      *
      * <P>Extended RAM from the Mapper Hardware</P>
      *
      */
      
      protected int mapper_extram[]   = new int[0x10000];
      
      
      
     /**
      *
      * <P>Size of Extended RAM from Mapper</P>
      *
      */
      
      public int mapper_extramsize = 0;



     /**
      *
      * <P>Respect Save RAM Writes.</P>
      *
      */

      public boolean enableSaveRAM = true;



     /**
      *
      * <P>The Program ROM Bank addresses.</P>
      *
      */

      private int bank[] = new int[16];  // Each bank is 0x1000



     /**
      *
      * <P>True if Save RAM was loaded with the Cart.</P>
      *
      */

      private boolean saveRAMWasLoaded = false;



#ifdef STAND

     /**
      *
      * <P>The Game Genie Decoder.</P>
      *
      */

      protected GameGenie gameGenie = new GameGenie();


     /**
      *
      * <P>Whether the Game Genie is Active.</P>
      *
      */

      public boolean gameGenieActive = false;


#endif


     /**
      *
      * <P>Variable for Zapper Trigger</P>
      *
      */

      public int zapperTrigger = 0;



     /**
      *
      * <P>Variable for Last Record Zapper X Position</P>
      *
      */

      public int zapperX = 0;



     /**
      *
      * <P>Variable for Last Record Zapper Y Position</P>
      *
      */

      public int zapperY = 0;



     /**
      *
      * <P>Is the NESCafe ROM Currently Loaded</P>
      *
      */

      private boolean nescaferomloaded = false;



     /**
      *
      * <P>Create a new Memory Manager.</P>
      *
      */

      public MemoryManager (NES nes, GUI gui) 
      {


         // Grab References to the GUI and NES

            this.gui = gui;
            this.nes = nes;
            this.ppu = nes.ppu;

      }



     /**
      *
      * <P>Return the current scanline number for Mappers.</P>
      *
      */

      public final int getScanline() 
      {

         return gui.tvController.scanLine;
      }



     /**
      *
      * <P>Initialise the Memory Manager.</P>
      *
      */

      public final void init(NESCart game, String fileName) 
      {


         // Fetch the Program ROM

            programROM = game.getProgROM();



#ifdef DECOMPILER

         // Clear Down Execute Tags
         
            programROMExecuteTags = new int[programROM.length];
            
            for (int i=0; i<programROMExecuteTags.length; i++)
               programROMExecuteTags[i] = -1;
            
            
#endif

         // Initialise the PPU Memory

            nes.ppu.PPUInit(game.getCharROM(), game.getMirroring(), game.getFourScreenNT());


         // Clear the Main Memory

            for (int i=0; i<memory.length; i++)
               memory[i] = 0;


         // Load the Trainer ROM if it Exists

            if (game.hasTrainer && game.trainerROM != null) 
            {

               for (int i=0; i<512; i++)
                  saveRAM[0x1000 + i] = game.trainerROM[i];

            } 
            else 
            {

               for (int i=0; i<512; i++)
                  saveRAM[0x1000 + i] = 0;

            }



         // Reset Frame IRQ Status
         
            nes.frameIRQEnabled = 0xFF;
            nes.frameIRQDisenabled = 0;
            
            
            
         // Load Save RAM if it Exists

            if (game.hasSaveRAM) 
            {

               try 
               {

                   // Assume Successful Load

                      saveRAMWasLoaded = true;


                   // Attempt to Load

                      if (fileName != null) loadSaveRAM(fileName);

               }
               catch (Exception e) {}


            } 
            else 
            {

               saveRAMWasLoaded = false;

            }
            

#ifndef APPLET

         // Clear any Game Genie Codes

            gameGenie.clearCodes();
            gameGenieActive = false;

#endif



         // Determine if this is the NESCafe Demo ROM

            StringBuffer sb = new StringBuffer();

            for (int i=0; i<8; i++)
               sb.append((char)programROM[i]);



         // Check for NESCafe Signature

            nescaferomloaded = sb.toString().equals("NESCafe ");



      }


     /**
      *
      * <P>Method to Load the Save RAM for the current Cartridge.</P>
      *
      */

      public final void loadSaveRAM(String cartName) throws Exception 
      {



            // Attempt to Create Save Directory

               File saveramDirectory = new File("saveram");


            // Construct the Filename

               String fileName = DavOS.getFileName(cartName) + ".sav";


            // Check if Exists and Load

               File saveRAMFile = new File(saveramDirectory + "/" + fileName);


               if (saveRAMFile.exists() && saveRAMFile.isFile())
               {


                  // Read File

                     FileInputStream saveFile = new FileInputStream(saveRAMFile);


                  // Load the Data into SaveRAM

                     for (int i = 0; i < saveRAM.length; i++)
                        saveRAM[i] = saveFile.read() & 0xFF;


                  // Close the File

                     saveFile.close();


               }
               else
               {


                  // Clear the SaveRAM

                     for (int i = 0; i < saveRAM.length; i++)
                        saveRAM[i] = 0;


               }


      }


     /**
      *
      * <P>Method to Save the Save RAM from the current Cartridge.</P>
      *
      */

      public final void saveSaveRAM() 
      {


         // Check that SaveRAM was Loaded Originally

            if (!saveRAMWasLoaded) return;


            try 
            {


               // Attempt to Create Save Directory
   
                  File saveramDirectory = new File("saveram");

                  if (saveramDirectory.exists() && !saveramDirectory.isDirectory())
                  {
                     saveramDirectory.delete();
                  }

                  if (!saveramDirectory.exists())
                  {
                     saveramDirectory.mkdir();
                  }


               // Construct the Filename

                  String fileName = saveramDirectory + "/" + DavOS.getFileName(nes.currentCartFileName) + ".sav";
                  FileOutputStream saveFile = new FileOutputStream(fileName);



               // Write the SaveRAM to Disk

                  for (int i = 0; i < saveRAM.length; i++)
                        saveFile.write( saveRAM[i] & 0xFF );


               // Close the File

                  saveFile.close();



            } catch (IOException e) {}


      }


     /**
      *
      * <P>Sets the offset in Program ROM of a Memory Bank.</P>
      *
      * @param bankNum The bank number to configure (0-15).
      * @param offsetInPRGROM The offset in Program ROM that the bank starts at.
      *
      */

      public final void setBankStartAddress (int bankNum, int offsetInPRGROM) 
      {

         offsetInPRGROM %= programROM.length;
         bank[bankNum % bank.length] = offsetInPRGROM;

      }


     /**
      *
      * <P>Read from Memory.</P>
      *
      * @return The value at the specified address.
      *
      */

      public final int read(int addr) 
      {



#ifdef PUNCHOUT
#ifdef APPLET


         // Do Adjustments for Special PunchOut Modes
            
            if (nes.applet!=null && nes.applet.PunchoutSpecialMode != "")
            {
               
               // Special 0 (Low Energy)
            
                  if (nes.applet.PunchoutSpecialMode.equals("0"))
                  {
                  
                     if (memory[0x0393]>5)
                        memory[0x0391] = 5;
                  
                  }
                  
               
               // Can't Get Up Again!
               
                  else if (nes.applet.PunchoutSpecialMode.equals("1"))
                  {

                      if (memory[0x008F]<2)
                      {
                      
                        memory[0x008F] = 2;
                      }
                            
                                        
                  }
   
   
   
               // Fast Rounds
               
                  else if (nes.applet.PunchoutSpecialMode.equals("2"))
                  {
      
                     memory[0x308] = 16;
      
                  
                  }
   
   
               // No More Stars
               
                  else if (nes.applet.PunchoutSpecialMode.equals("3"))
                  {
         
                     
                     memory[0x0341] = 0;
                     
                  
                  }
   
   
               // Invincible Mode
               
                  else if (nes.applet.PunchoutSpecialMode.equals("4"))
                  {
      
      
                     memory[0x0341] = 3;      // Stars
                     memory[0x0321] = 9;      // Hearts MSB
                     memory[0x0322] = 9;      // Hearts LSB
                     memory[0x0325] = 0x80;   // Hearts Redraw Flag
                     memory[0x0391] = 0x60;   // Energy for Mac to Max
                     
                     memory[0x008F] = 0;      // TKO Count to 0
                     memory[0x03D0] = 2;      // Mac Down 0 in Total
                     memory[0x03DD] = 2;      // Mac Down 0 in R1
                     memory[0x03DE] = 2;      // Mac Down 0 in R2
                     memory[0x03DF] = 2;      // Mac Down 0 in R3

                     memory[0x03C1] = 0;      // Keep 0 for Infinite Knock-Downs
                   
                     
                  }
   
   
   
            }

#endif
#endif 





         if (addr < 0x2000) 
         { 


            // RAM Mirrored 4 Times

               return memory[addr & 0x7FF];

         }
         else if (addr < 0x4000) 
         { 
         
            // Input/Output

               return nes.ppu.read(addr & 0xE007);

         }
         else if (addr < 0x4016) 
         {
            
            
            // SPR-RAM DMA
            
               if (addr == 0x4014)
               {
                  
                     return 0;
                     
               }
               else if (addr == 0x4015 && ((nes.frameIRQEnabled & 0xC0)==0))
               {

#ifdef SOUND

                  return (nes.sound != null) ? nes.sound.read(addr) | 0x40 : 0x40;

#else

                  return 40;

#endif

               }   
         
            
#ifdef SOUND

            // Read from Sound Chip

               return (nes.sound != null) ? nes.sound.read(addr) : 0;

#else

               return 0;
               
#endif
    
            
         }
         else if (addr < 0x4018) 
         { 
         
            // High I/O Regs

               if (addr == 0x4016)
               {
                  
                  // Joypad #1
                  
                     return nes.joyPad1.readJoyPadBit() | readZapperData(1);
       
               }
               else if (addr == 0x4017)
               {
                  
                  // Joypad #2
   
                     return nes.joyPad2.readJoyPadBit() | readZapperData(2);
                     
               }
              
               return 0;

         }
         else if (addr < 0x6000) 
         { 
         
            // Expansion ROM

               return nes.mapper.accesslowread(addr);


         }
         else if (addr < 0x8000) 
         { 
         
            // 0x6000 - 0x7FFF SaveRAM

               return saveRAM[addr - 0x6000];


         } 
         else 
         {

#ifndef APPLET

            // Check if Game Genie is Enabled

               if (gameGenieActive)
               {


                  // Check if a Match was Found in the Game Genie Decoder

                     int matchIndex = gameGenie.addressMatch(addr);


                  // Grab Information from the Matched Code

                     if (matchIndex >= 0) 
                     {


                        // Read the Compare Value

                           int compareVal = gameGenie.getValue(matchIndex);

                        // If no Compare Value then Trigger Code

                           if (compareVal == -1) return gameGenie.getValue(matchIndex);


                        // Read the Memory Address

                           int b = bank [ ((addr & 0xF000) >> 12)];
                           int actualVal = programROM[ b + (addr & 0x0FFF) ];


                        // Trigger the Game Genie if Compare Value doesn't Match Real Value

                           if (actualVal == compareVal) 
                           {

                              return actualVal;

                           } 
                           else 
                           {

                              return gameGenie.getValue(matchIndex);

                           }


                     }

               }
               
#endif


            // Read the Memory Address

               try 
               {


                  // Get Offset

                     int c = bank[ ((addr & 0xF000) >> 12) ];


                  // Return Memory

                     return programROM[ c + (addr & 0x0FFF) ];


               } 
               catch (Exception e) { return 0; }

         }


      }



#ifdef DEBUG


     /**
      *
      * Output File Used for Developer Debugging of Memory
      *
      */
      
      private FileOutputStream outputDebug = null;
      
    
     /**
      *
      * Method used for Developer Debugging
      *
      */
  
      public void debugOutput(String line)
      {
         
         try
         {
            
            if (outputDebug != null)
               outputDebug.write(line.getBytes());

         }
         catch (Exception e)
         {
            
         }
      }
      
      
      
     /**
      *
      * Method used for Developer Debugging
      *
      */
  
      public void debugDump()
      {
         for (int i=0; i<0x3FF; i++)
            debugOutput("0x" + DavOS.hex(i,4) + "," + memory[i] + ",\r\n");
            
      }
      
      
      
     /**
      *
      * Method used for Developer Debugging
      *
      */
  
      public void debugOpen()
      {

         try
         {
            
            outputDebug = new FileOutputStream (new File("debug.csv"));
            
         }
         catch (Exception e)
         {
            outputDebug = null;
         }         
         
      }
      
      
      
     /**
      *
      * Method used for Developer Debugging
      *
      */
  
      public void debugClose()
      {
         try
         {
         
            outputDebug.close();
            outputDebug = null;

         }
         catch (Exception e)
         {}
                  
      }
      
      
#endif



#ifdef PUNCHOUT

     /**
      *
      * <P>Broadcast Success Failure in Punchout Special Edition</P>
      *
      */

      public boolean broadCastPunchoutData(boolean win)
      {
         

 
#ifdef SOUND
      
            // Disable the Tone
            
               nes.sound.enableSound(false);

#endif


            // Halt the CPU and Show Logo
            
               nes.cpu.haltCPU();
               nes.gui.showNESCafeLogo();
         
      

               
            // Grab Score Details

               try
               {
               

                  // Grab Score Details from the Game
                  
                     int sBoxer = memory[0x0001];
                     int sMeDn  = memory[0x03D0];
                     int sBadDn = memory[0x03D1];
                     
                     int sRound = memory[0x0006];
                     int sTime  = memory[0x0302] * 100 + memory[0x0304] * 10 + memory[0x0305];
                     int sMac   = memory[0x0391];
                     int sBad   = memory[0x039A];
                     int sWin   = (win) ? 1: 0;
                     
                     
                     
                     
                  // Determine if Mr Dream or Mike Tyson Version
                  
                     long crc = nes.cpu.currentCart.crc32;
                     
                     if (crc==3197345742l)
                     {
                        // Mr Dream Punch Out
                        
                        
                     }
                     if (crc==2460096604l)
                     {
                        // Mike Tyson Punch Out
                        
                           if (sBoxer==13)
                              sBoxer++;
                              
                     }
                     
                     
                     
                  // Check if Java Version Supports UTF-8 Encoding
                  
                     boolean oldJavaVersion = false;
                     
                     try
                     {
                        String test = URLEncoder.encode("A","UTF-8");
                     }
                     catch (NoSuchMethodError nsme)
                     {
                        oldJavaVersion = true;
                     }
                     
                     
                     
                  // Compile the Post Data
                     
                     String data = "";
                     
                     if (oldJavaVersion)
                     {
                        
                            data = "round="    + URLEncoder.encode(sRound + "") + "&"
                                 + "time="     + URLEncoder.encode(sTime  + "") + "&"
                                 + "mac="      + URLEncoder.encode(sMeDn  + "") + "&"
                                 + "bad="      + URLEncoder.encode(sBadDn + "") + "&"
                                 + "win="      + URLEncoder.encode(sWin   + "") + "&"
                                 + "boxer="    + URLEncoder.encode(sBoxer + "");
                                 

                           // Add User if Applet
                           

#ifdef APPLET
                              
                              data+= "&user=" + URLEncoder.encode(nes.applet.username);
                              

#endif
                                      
                                      
                     }
                     else
                     {
                     
                           // Store Data
                           
                            data = "round="    + URLEncoder.encode(sRound + "", "UTF-8") + "&"
                                 + "time="     + URLEncoder.encode(sTime  + "", "UTF-8") + "&"
                                 + "mac="      + URLEncoder.encode(sMeDn  + "", "UTF-8") + "&"
                                 + "bad="      + URLEncoder.encode(sBadDn + "", "UTF-8") + "&"
                                 + "win="      + URLEncoder.encode(sWin   + "", "UTF-8") + "&"
                                 + "boxer="    + URLEncoder.encode(sBoxer + "", "UTF-8");
                                 

                              
                           // Add User if Applet

#ifdef APPLET
                              
                              data+= "&user=" + URLEncoder.encode(nes.applet.username,"UTF-8");
                                 
#endif
                                          
                                 
                     }
                         
                                                            
                  // OK, now you know hows its done please do not use this Knowledge to improve your score!
                  
                     return ServerConnection.postData(data);
                  
                  
               }
               catch (Exception e) 
               {
                  e.printStackTrace();
                  
               }


               return false;
               
         
      }
      

#endif




     /**
      *
      * <P>Write to Memory.</P>
      *
      */

      public final void write(int addr, int value) 
      {


         if (addr < 0x2000) 
         {  




#ifdef PUNCHOUT

                  switch (addr)
                  {
                     case 0x0170:
                     case 0x0171:
                     case 0x0172:
                     case 0x0173:
                     case 0x0174:
                     case 0x0175:
                     {



#ifdef PUNCHOUT                                 

                        // Only Broadcast if Allowed
                        
                           if (nes.applet.PunchoutSubmitEnabled)
                           {

                                
     
                              // Punchout Wins Stored in 0x170,0x171 (HL)
         
                                 if ((addr == 0x171 && value != memory[addr] && value !=0) || (addr == 0x170 && value != memory[addr] && value != 0))
                                 {

                                    // Check if User Cheated
                                    
                                       if (nes.applet!= null && nes.applet.PunchoutCheatsEnabled)
                                       {

                                          // Explain that Cheats dont Prosper
                                           
                                             nes.gui.writeToTopBar("Well done, but scores in Cheat Mode are not kept.");

#ifdef SOUND
      
                                          // Disable the Tone
                                          
                                             nes.sound.enableSound(false);

#endif

                                          // Halt the CPU and Show Logo
                                          
                                             nes.cpu.haltCPU();
                                             nes.gui.showNESCafeLogo();
                                             return;
                                             
                                       }
               
      
      
                                    // Broadcast
      
                                       nes.gui.writeToTopBar("Please wait while your score is submitted...");
                                       if (broadCastPunchoutData(true))
                                       {
                                          nes.gui.writeToTopBar("Well done, your win has been submitted!");
                                       }
                                       else               
                                       {
                                          nes.gui.writeToTopBar("Well done, but your score failed to submit.");
                                       }
                                       return;
                                 
                                 
                                 }
                                 
                                 
                              // Punchout Loses Stored in 0x172,0x173 (HL)
         
                                 if ((addr == 0x173 && value != memory[addr] && value !=0) || (addr == 0x172 && value != memory[addr] && value != 0))
                                 {
                             
                                    // Check if User Cheated
                                    
                                       if (nes.applet!= null && nes.applet.PunchoutCheatsEnabled)
                                       {
                                           
#ifdef SOUND
      
                                          // Disable the Tone
                                          
                                             nes.sound.enableSound(false);

#endif

                                          // Halt the CPU and Show Logo
                                          
                                             nes.cpu.haltCPU();
                                             nes.gui.showNESCafeLogo();
                                             return;

                                       }



                                    // Broadcast

                                       nes.gui.writeToTopBar("Please wait while your score is submitted...");
       
                                       if (broadCastPunchoutData(false))
                                       {
                                          nes.gui.writeToTopBar("Bad luck, your loss has been submitted!");
                                       }
                                       else               
                                       {
                                          nes.gui.writeToTopBar("Bad luck, fortunately your score failed to submit.");
                                       }
                                       return;
                                    
                                 }
                           


                        }
                        
#endif

        
                        break;
                    
                     }
                      
                     default:
                     {
                        break;
                     }
                                          
                  }
       
            
#endif


            // 0x0000 - 0x1FFF RAM

               memory[addr & 0x7FF] = value;
               return;

         }

         else if(addr < 0x4000) 
         { 
         
            // Low IO Registers
               
               nes.ppu.write(addr & 0xE007,value);
               return;

         }

         else if(addr < 0x4018) 
         { 
         
            // High IO Registers

              nes.mapper.accesslow(addr,value);
              
              switch(addr) 
              {

              case 0x4000:
              case 0x4001:
              case 0x4002:
              case 0x4003:
              case 0x4004:
              case 0x4005:
              case 0x4006:
              case 0x4007:
              case 0x4008:
              case 0x4009:
              case 0x400A:
              case 0x400B:
              case 0x400C:
              case 0x400D:
              case 0x400E:
              case 0x400F:
              case 0x4010:
              case 0x4011:
              case 0x4012:
              case 0x4013:


#ifdef SOUND

                   if (nes.sound != null) nes.sound.write(addr,value);
                   
#endif

                   return;

              case 0x4014: // Sprite DMA Register

                   int source[] = memory;
                   int k = value << 8;


                   switch (k & 0xF000) 
                   {

                      case 0x8000: // DMA Transfer from Program ROM
   
                           source = programROM;
                           k = bank[ ( k >> 12) + (k & 0x0FFF) ];
                           break;
   
   
                      case 0x6000: // DMA Transfer from SaveRAM
                      case 0x7000:

#ifdef DEBUG

                           System.out.println("Sprite DMA Attempted from SaveRAM");
                           
#endif

                           return;
   
   
                      case 0x5000: // DMA Transfer from Expansion RAM

#ifdef DEBUG

                           System.out.println("Sprite DMA Attempted from Expansion RAM");
                           
#endif

                           return;
   
   
                      case 0x2000: // DMA Transfer from Registers
                      case 0x3000:
                      case 0x4000:
   
                           return;
   
   
                      case 0x0000: // DMA Transfer from RAM
                      case 0x1000:
   
                           source = memory;
                           k &= 0x7FF;
                           break;

                   }

                  
                  // Perform the DMA Transfer
                  
                     for (int i = 0; i < 256; i++) 
                     {
                     
                        nes.ppu.spriteMemory[i] = source[k] & 0xFF;
                        k++;
                        
                     }
                     
   

                  // Burn Some Cycles
                  
                     nes.cpu.eatCycles(514);
                     return;



              case 0x4015:

#ifdef SOUND

                   if (nes.sound != null) nes.sound.write(addr,value);

#endif

                   return;


              case 0x4016: // Joypad #1

                   if ((value & 0x1) == 0) nes.joyPad1.resetJoyPad();
                   return;
            
              case 0x4017: // Joypad #2

                   if ((value & 0x1) == 0) nes.joyPad2.resetJoyPad();

                   if (nes.frameIRQDisenabled==0)
                   {
                   
                     nes.frameIRQEnabled = value;
                     
                   }
                   
#ifdef SOUND

                   if (nes.sound != null) nes.sound.write(addr,value);
#endif
                   
                   return;


              }

              return;

         }


         else if(addr < 0x6000) 
         { 
         
            // Expansion ROM and Low Mapper Write Region
               
               nes.mapper.accesslow(addr,value);

         }

         else if(addr < 0x8000) 
         { 
         
            // Save RAM
            
               nes.mapper.access(addr,value);
               if (enableSaveRAM) saveRAM[addr - 0x6000] = value;
               return;

         }
         
         else 
         {

              nes.mapper.access(addr,value);

         }


      }



     /**
      *
      * <P>Read a 16 bit Word from Memory.</P>
      *
      */

      public final int readWord(int address) 
      {

         return read(address) | (read(address + 1) << 8);

      }


     /**
      *
      * <P>Write a 16 bit Word to Memory.</P>
      *
      */

      public final void writeWord(int address, int value) 
      {

         write(address, value & 0xFF);
         write(address + 1, value >> 8);

      }





     /**
      *
      * <P>Returns the Program ROM.</P>
      *
      */

      public final int[] getProgramROM() 
      {

         return programROM;

      }



     /**
      * 
      * <P>Loads the State of the Memory from an InputStream.</P>
      *
      */

      public final void stateLoad(java.io.InputStream input) throws java.io.IOException 
      {

         // Load Memory

            for (int i=0; i<memory.length; i++)
               memory[i] = input.read() & 0xFF;


         // Load SaveRAM

            for (int i=0; i<saveRAM.length; i++)
               saveRAM[i] = input.read() & 0xFF;


         // Load Bank Addresses

            for (int i=0; i<bank.length; i++) 
            {

               bank[i]  = (input.read() & 0xFF) << 0x00;
               bank[i] |= (input.read() & 0xFF) << 0x08;
               bank[i] |= (input.read() & 0xFF) << 0x10;
               bank[i] |= (input.read() & 0xFF) << 0x18;
  
            }


         // Load Emulation Information

            enableSaveRAM    = (input.read() == 0xFF);
            saveRAMWasLoaded = (input.read() == 0xFF);

      }



     /**
      * 
      * <P>Saves the State of the Memory to a FileOutputStream.</P>
      *
      */

      public final void stateSave(java.io.OutputStream output) throws java.io.IOException 
      {


         // Save Memory

            for (int i=0; i<memory.length; i++)
               output.write(memory[i] & 0xFF);


         // Save SaveRAM

            for (int i=0; i<saveRAM.length; i++)
               output.write(saveRAM[i] & 0xFF);


         // Save Bank Addresses

            for (int i=0 ; i< bank.length; i++) 
            {

               output.write( (bank[i] >> 0x00) & 0xFF);
               output.write( (bank[i] >> 0x08) & 0xFF);
               output.write( (bank[i] >> 0x10) & 0xFF);
               output.write( (bank[i] >> 0x18) & 0xFF);

            }

    
         // Emulation Information

            output.write(enableSaveRAM    ? 0xFF : 0x00);
            output.write(saveRAMWasLoaded ? 0xFF : 0x00);
            
      }



     /**
      * 
      * <P>Read Zapper Values.</P>
      *
      */

      private int readZapperData(int portNum)
      {


         if ( nes.usersettings.controllerZapperEnabled )
         {

            // Grab Pixel

               int color = gui.tvController.videoBuffer[zapperY*256+zapperX];
               int retVal = zapperTrigger + 0x08;


            // Check for Palette Entries 32 or 48 (White)

               if (color == -1)
                  return retVal & 0xF7;				

               return retVal;

         }

         return 0;


      }


     /**
      *
      * Check for NESCafe ROM
      *
      */

      public boolean isNESCafeROM()
      {

         // Return whether NESCafe ROM Loaded - Determined on Init

            return nescaferomloaded;


      }


}

#endif
