/**
 * Mupen64 - memory.c
 * Copyright (C) 2002 Hacktarux
 *
 * Mupen64 homepage: http://mupen64.emulation64.com
 * email address: hacktarux@yahoo.fr
 * 
 * If you want to contribute to the project please contact
 * me first (maybe someone is already making what you are
 * planning to do).
 *
 *
 * This program is free software; you can redistribute it and/
 * or modify it under the terms of the GNU General Public Li-
 * cence as published by the Free Software Foundation; either
 * version 2 of the Licence, or any later version.
 *
 * This program is distributed in the hope that it will be use-
 * ful, but WITHOUT ANY WARRANTY; without even the implied war-
 * ranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public Licence for more details.
 *
 * You should have received a copy of the GNU General Public
 * Licence along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
**/

#include <stdio.h>
#include <stdlib.h>

#include "memory.h"
#include "dma.h"
#include "../r4300/r4300.h"
#include "../r4300/macros.h"
#include "../r4300/interupt.h"
#include "../r4300/recomph.h"
#include "../sound/sound.h"
#include "pif.h"
#include "flashram.h"
#ifndef __WIN32__
#include "../main/winlnxdefs.h"
#else
#include <windows.h>
#endif
#include "../main/plugin.h"

/* definitions of the rcp's structures and memory area */
RDRAM_register rdram_register;
mips_register MI_register;
PI_register pi_register;
SP_register sp_register;
SI_register si_register;
VI_register vi_register;
RI_register ri_register;
AI_register ai_register;
DPC_register dpc_register;
unsigned long rdram[0x800000/4];
unsigned char *rdramb = (unsigned char *)(rdram);
unsigned long SP_DMEM[0x1000/4];
unsigned char *SP_DMEMb = (unsigned char *)(SP_DMEM);
unsigned long SP_IMEM[0x1000/4];
unsigned char *SP_IMEMb = (unsigned char *)(SP_IMEM);
unsigned long PIF_RAM[0x40/4];
unsigned char *PIF_RAMb = (unsigned char *)(PIF_RAM);
unsigned long RSP_PC_REG;

// address : address of the read/write operation being done
unsigned long address = 0;
// *address_low = the lower 16 bit of the address :
#ifdef _BIG_ENDIAN
static unsigned short *address_low = (unsigned short *)(&address)+1; 
#else
static unsigned short *address_low = (unsigned short *)(&address);
#endif

// values that are being written are stored in these variables
unsigned long word;
unsigned char byte;
unsigned short hword;
unsigned long long int dword;

// addresse where the read value will be stored
unsigned long long int* rdword;

// trash : when we write to unmaped memory it is written here
static unsigned long trash;

// hash tables of read functions
void (*readmem[0xFFFF])();
void (*readmemb[0xFFFF])();
void (*readmemh[0xFFFF])();
void (*readmemd[0xFFFF])();

// hash tables of write functions
void (*writemem[0xFFFF])();
void (*writememb[0xFFFF])();
void (*writememd[0xFFFF])();
void (*writememh[0xFFFF])();

// memory sections
static unsigned long *readrdramreg[0xFFFF];
static unsigned long *readrspreg[0xFFFF];
static unsigned long *readrsp[0xFFFF];
static unsigned long *readmi[0xFFFF];
static unsigned long *readvi[0xFFFF];
static unsigned long *readai[0xFFFF];
static unsigned long *readpi[0xFFFF];
static unsigned long *readri[0xFFFF];
static unsigned long *readsi[0xFFFF];
static unsigned long *readdp[0xFFFF];


int init_memory()
{
   int i;
   
   //swap rom
   unsigned long *roml;
   roml = (void *)rom;
   for (i=0; i<(taille_rom/4); i++) roml[i] = sl(roml[i]);
   
   //init hash tables
   for (i=0; i<(0x10000); i++)
     {
	readmem[i] = read_nomem;
	readmemb[i] = read_nomemb;
	readmemd[i] = read_nomemd;
	readmemh[i] = read_nomemh;
	writemem[i] = write_nomem;
	writememb[i] = write_nomemb;
	writememd[i] = write_nomemd;
	writememh[i] = write_nomemh;
     }
   
   //init RDRAM
   for (i=0; i<(0x800000/4); i++) rdram[i]=0;
   for (i=1; i<0x80; i++) 
     {
	writemem[(0xa000+i)] = write_rdram;
	writemem[(0x8000+i)] = write_rdram;
	writememb[(0xa000+i)] = write_rdramb;
	writememb[(0x8000+i)] = write_rdramb;
	writememh[(0xa000+i)] = write_rdramh;
	writememh[(0x8000+i)] = write_rdramh;
	writememd[(0x8000+i)] = write_rdramd;
	writememd[(0xa000+i)] = write_rdramd;
     }
   for (i=0; i<0x80; i++)
     {
	readmem[(0xa000+i)] = read_rdram;
	readmem[(0x8000+i)] = read_rdram;
	readmemb[(0xa000+i)] = read_rdramb;
	readmemb[(0x8000+i)] = read_rdramb;
	readmemh[(0xa000+i)] = read_rdramh;
	readmemh[(0x8000+i)] = read_rdramh;
	readmemd[(0xa000+i)] = read_rdramd;
	readmemd[(0x8000+i)] = read_rdramd;
     }
   writemem[0xa000] = write_rdram_int;
   writemem[0x8000] = write_rdram_int;
   writememb[0xa000] = write_rdram_intb;
   writememb[0x8000] = write_rdram_intb;
   writememh[0xa000] = write_rdram_inth;
   writememh[0x8000] = write_rdram_inth;
   writememd[0xa000] = write_rdram_intd;
   writememd[0x8000] = write_rdram_intd;
   
   for (i=0x80; i<0x3F0; i++) writemem[0x8000+i] = write_nothing;
   for (i=0x80; i<0x3F0; i++) readmem[0x8000+i] = read_nothing;
   for (i=0x80; i<0x3F0; i++) readmemh[0x8000+i] = read_nothingh;
   
   //init RDRAM registers
   writemem[0xa3F0] = write_rdramreg;
   readmem[0xa3F0] = read_rdramreg;
   writemem[0xa3F8] = write_nothing;
   for (i=0x28; i<0x10000; i++) readrdramreg[i] = &trash;
   rdram_register.rdram_mode=0;
   readrdramreg[0xC] = &rdram_register.rdram_mode;
   
   //init RSP resgisters
   readmem[0xa400] = read_rsp_mem;
   readmemb[0xa400] = read_rsp_memb;
   writemem[0xa400] = write_rsp_mem;
   writememb[0xa400] = write_rsp_memb;
   writememd[0xa400] = write_rsp_memd;
   for (i=0; i<(0x1000/4); i++) SP_DMEM[i]=0;
   for (i=0; i<(0x1000/4); i++) SP_IMEM[i]=0;
   readmem[0xa404] = read_rsp_reg;
   writemem[0xa404] = write_rsp_reg;
   sp_register.sp_mem_addr_reg=0;
   readrspreg[0x0] = &sp_register.sp_mem_addr_reg;
   sp_register.sp_dram_addr_reg=0;
   readrspreg[0x4] = &sp_register.sp_dram_addr_reg;
   sp_register.sp_rd_len_reg=0;
   readrspreg[0x8] = &sp_register.sp_rd_len_reg;
   sp_register.sp_wr_len_reg=0;
   readrspreg[0xc] = &sp_register.sp_wr_len_reg;
   sp_register.sp_status_reg=1;
   readrspreg[0x10] = &sp_register.sp_status_reg;
   sp_register.w_sp_status_reg=0;
   sp_register.halt=1;
   sp_register.broke=0;
   sp_register.single_step=0;
   sp_register.intr_break=0;
   sp_register.signal0=0;
   sp_register.signal1=0;
   sp_register.signal2=0;
   sp_register.signal3=0;
   sp_register.signal4=0;
   sp_register.signal5=0;
   sp_register.signal6=0;
   sp_register.signal7=0;
   sp_register.sp_dma_full_reg=0;
   readrspreg[0x14] = &sp_register.sp_dma_full_reg;
   sp_register.sp_dma_busy_reg=0;
   readrspreg[0x18] = &sp_register.sp_dma_busy_reg;
   sp_register.sp_semaphore_reg=0;
   readrspreg[0x1C] = &sp_register.sp_semaphore_reg;
   readmem[0xa408] = read_rsp;
   writemem[0xa408] = write_rsp;
   //RSP_PC_REG=0x4001000;
   RSP_PC_REG=0;
   readrsp[0x0] = &RSP_PC_REG;
   
   //init rdp command registers
   writemem[0xa410] = write_dp;
   readmem[0xa410] = read_dp;
   for (i=0; i<(0x10000); i++) readdp[i] = NULL;
   dpc_register.dpc_start=0;
   readdp[0x0] = &dpc_register.dpc_start;
   dpc_register.dpc_end=0;
   readdp[0x4] = &dpc_register.dpc_end;
   dpc_register.dpc_current=0;
   dpc_register.w_dpc_status=0;
   dpc_register.dpc_status=0;
   readdp[0xC] = &dpc_register.dpc_status;
   dpc_register.xbus_dmem_dma=0;
   dpc_register.freeze=0;
   dpc_register.dpc_clock=0;
   dpc_register.dpc_bufbusy=0;
   dpc_register.dpc_pipebusy=0;
   dpc_register.dpc_tmem=0;
   
   //init mips registers
   writemem[0xa430] = write_mi;
   readmem[0xa430] = read_mi;
   for (i=0; i<(0x10000); i++) readmi[i] = NULL;
   MI_register.w_mi_init_mode_reg = 0;
   MI_register.mi_init_mode_reg = 0;
   readmi[0x0] = &MI_register.mi_init_mode_reg;
   MI_register.mi_version_reg = 0x02020102;
   //MI_register.mi_version_reg = 0x01010101;
   readmi[0x4] = &MI_register.mi_version_reg;
   MI_register.mi_intr_reg = 0;
   readmi[0x8] = &MI_register.mi_intr_reg;
   MI_register.mi_intr_mask_reg = 0;
   readmi[0xC] = &MI_register.mi_intr_mask_reg;
   MI_register.w_mi_intr_mask_reg = 0;
   MI_register.init_length = 0;
   MI_register.init_mode = 0;
   MI_register.ebus_test_mode = 0;
   MI_register.RDRAM_reg_mode = 0;
   MI_register.SP_intr_mask = 0;
   MI_register.SI_intr_mask = 0;
   MI_register.AI_intr_mask = 0;
   MI_register.VI_intr_mask = 0;
   MI_register.PI_intr_mask = 0;
   MI_register.DP_intr_mask = 0;
   
   //init PI registers
   readmem[0xa460] = read_pi;
   writemem[0xa460] = write_pi;
   for (i=0; i<(0x10000); i++) readpi[i] = NULL;
   pi_register.pi_dram_addr_reg = 0;
   readpi[0x0] = &pi_register.pi_dram_addr_reg;
   pi_register.pi_cart_addr_reg = 0;
   readpi[0x4] = &pi_register.pi_cart_addr_reg;
   pi_register.pi_rd_len_reg = 0;
   readpi[0x8] = &pi_register.pi_rd_len_reg;
   pi_register.pi_wr_len_reg = 0;
   readpi[0xC] = &pi_register.pi_wr_len_reg;
   pi_register.read_pi_status_reg = 0;
   readpi[0x10] = &pi_register.read_pi_status_reg;
   pi_register.pi_bsd_dom1_lat_reg = 0;
   readpi[0x14] = &pi_register.pi_bsd_dom1_lat_reg;
   pi_register.pi_bsd_dom1_pwd_reg = 0;
   readpi[0x18] = &pi_register.pi_bsd_dom1_pwd_reg;
   pi_register.pi_bsd_dom1_pgs_reg = 0;
   readpi[0x1c] = &pi_register.pi_bsd_dom1_pgs_reg;
   pi_register.pi_bsd_dom1_rls_reg = 0;
   readpi[0x20] = &pi_register.pi_bsd_dom1_rls_reg;
   pi_register.pi_bsd_dom2_lat_reg = 0;
   readpi[0x24] = &pi_register.pi_bsd_dom2_lat_reg;
   pi_register.pi_bsd_dom2_pwd_reg = 0;
   readpi[0x28] = &pi_register.pi_bsd_dom2_pwd_reg;
   pi_register.pi_bsd_dom2_pgs_reg = 0;
   readpi[0x2c] = &pi_register.pi_bsd_dom2_pgs_reg;
   pi_register.pi_bsd_dom2_rls_reg = 0;
   readpi[0x30] = &pi_register.pi_bsd_dom2_rls_reg;
   
   //init RI registers
   readmem[0xa470] = read_ri;
   writemem[0xa470] = write_ri;
   for (i=0; i<(0x10000); i++) readri[i] = NULL;
   ri_register.ri_mode = 0;
   readri[0x0] = &ri_register.ri_mode;
   ri_register.ri_config = 0;
   readri[0x4] = &ri_register.ri_config;
   ri_register.ri_current_load = 0;
   readri[0x8] = &ri_register.ri_current_load;
   ri_register.ri_select = 0;
   readri[0xC] = &ri_register.ri_select;
   ri_register.ri_refresh = 0;
   readri[0x10] = &ri_register.ri_refresh;
   
   //init SI registers
   writemem[0xa480] = write_si;
   readmem[0xa480] = read_si;
   for (i=0; i<(0x10000); i++) readsi[i] = NULL;
   si_register.si_dram_addr = 0;
   readsi[0x0] = &si_register.si_dram_addr;
   si_register.si_pif_addr_rd64b = 0;
   readsi[0x4] = &si_register.si_pif_addr_rd64b;
   si_register.si_pif_addr_wr64b = 0;
   readsi[0x10] = &si_register.si_pif_addr_wr64b;
   si_register.si_status = 0;
   readsi[0x18] = &si_register.si_status;
   
   //init AI registers
   writemem[0xa450] = write_ai;
   readmem[0xa450] = read_ai;
   for (i=0; i<(0x10000); i++) readai[i] = NULL;
   ai_register.ai_dram_addr = 0;
   readai[0x00] = &ai_register.ai_dram_addr;
   ai_register.ai_len = 0;
   readai[0x04] = &ai_register.ai_len;
   ai_register.ai_control = 0;
   readai[0x08] = &ai_register.ai_control;
   ai_register.ai_status = 0;
   readai[0x0C] = &ai_register.ai_status;
   ai_register.ai_dacrate = 0;
   readai[0x10] = &ai_register.ai_dacrate;
   ai_register.ai_bitrate = 0;
   readai[0x14] = &ai_register.ai_bitrate;
   
   //init VI registers
   writemem[0xa440] = write_vi;
   readmem[0xa440] = read_vi;
   for (i=0; i<(0x10000); i++) readvi[i] = NULL;
   vi_register.vi_status = 0;
   readvi[0x0] = &vi_register.vi_status;
   vi_register.vi_origin = 0;
   readvi[0x4] = &vi_register.vi_origin;
   vi_register.vi_width = 0;
   readvi[0x8] = &vi_register.vi_width;
   vi_register.vi_v_intr = 0;
   readvi[0xC] = &vi_register.vi_v_intr;
   vi_register.vi_current = 0;
   readvi[0x10] = &vi_register.vi_current;
   vi_register.vi_burst = 0;
   readvi[0x14] = &vi_register.vi_burst;
   vi_register.vi_v_sync = 0;
   readvi[0x18] = &vi_register.vi_v_sync;
   vi_register.vi_h_sync = 0;
   readvi[0x1C] = &vi_register.vi_h_sync;
   vi_register.vi_leap = 0;
   readvi[0x20] = &vi_register.vi_leap;
   vi_register.vi_h_start = 0;
   readvi[0x24] = &vi_register.vi_h_start;
   vi_register.vi_v_start = 0;
   readvi[0x28] = &vi_register.vi_v_start;
   vi_register.vi_v_burst = 0;
   readvi[0x2C] = &vi_register.vi_v_burst;
   vi_register.vi_x_scale = 0;
   readvi[0x30] = &vi_register.vi_x_scale;
   vi_register.vi_y_scale = 0;
   readvi[0x34] = &vi_register.vi_y_scale;
   
   //init sram area
   readmem[0xa500] = read_nothing;
   readmemh[0xa500] = read_nothingh;
   
   //init n64ddrive
   readmem[0xa600] = read_nothing;
   
   //init flashram
   readmem[0xa800] = read_flashram_status;
   writemem[0xa801] = write_flashram_command;
   
   //init rom area
   for (i=0; i<(taille_rom >> 16); i++) readmem[0xb000+i] = read_rom;
   for (i=(taille_rom >> 16); i<0xfc0; i++) readmem[0xb000+i] = read_nothing;
   for (i=(taille_rom >> 16); i<0xfc0; i++) readmem[0x9000+i] = read_nothing;
   for (i=(taille_rom >> 16); i<0xfc0; i++) readmemh[0x9000+i] = read_nothingh;
   for (i=(taille_rom >> 16); i<0xfc0; i++) writemem[0xb000+i] = write_nothing;
   
   //init PIF_RAM
   readmem[0xbfc0] = read_pif;
   readmemb[0xbfc0] = read_pifb;
   writemem[0xbfc0] = write_pif;
   for (i=0; i<(0x40/4); i++) PIF_RAM[i]=0;
   
   use_flashram = 0;
   init_flashram();
   printf("memory initialized\n");
   return 0;
}

void free_memory()
{
}

static void update_MI_init_mode_reg()
{
   MI_register.init_length = MI_register.w_mi_init_mode_reg & 0x7F;
   if (MI_register.w_mi_init_mode_reg & 0x80) 
     MI_register.init_mode = 0;
   if (MI_register.w_mi_init_mode_reg & 0x100)
     MI_register.init_mode = 1;
   if (MI_register.w_mi_init_mode_reg & 0x800)
     {
	MI_register.mi_intr_reg &= 0xFFFFFFDF;
	if (!dynacore && interpcore) check_interupt();
     }
   if (MI_register.w_mi_init_mode_reg & 0x1000)
     MI_register.RDRAM_reg_mode=0;
   if (MI_register.w_mi_init_mode_reg & 0x2000)
     MI_register.RDRAM_reg_mode=1;
   if (MI_register.w_mi_init_mode_reg & (~0x39FF))
     {
	printf("unhandled write in MI init mode reg\n");
	printf("%x\n",
	       (unsigned int)(MI_register.w_mi_init_mode_reg)&(~0x39FF));
	stop = 1;
     }
   MI_register.mi_init_mode_reg = ((MI_register.init_length) |
				   (MI_register.init_mode << 7) |
				   (MI_register.RDRAM_reg_mode << 9)
				   );
}

static void update_MI_intr_mask_reg()
{
   if (MI_register.w_mi_intr_mask_reg & 0x1)   MI_register.SP_intr_mask = 0;
   if (MI_register.w_mi_intr_mask_reg & 0x2)   MI_register.SP_intr_mask = 1;
   if (MI_register.w_mi_intr_mask_reg & 0x4)   MI_register.SI_intr_mask = 0;
   if (MI_register.w_mi_intr_mask_reg & 0x8)   MI_register.SI_intr_mask = 1;
   if (MI_register.w_mi_intr_mask_reg & 0x10)  MI_register.AI_intr_mask = 0;
   if (MI_register.w_mi_intr_mask_reg & 0x20)  MI_register.AI_intr_mask = 1;
   if (MI_register.w_mi_intr_mask_reg & 0x40)  MI_register.VI_intr_mask = 0;
   if (MI_register.w_mi_intr_mask_reg & 0x80)  MI_register.VI_intr_mask = 1;
   if (MI_register.w_mi_intr_mask_reg & 0x100) MI_register.PI_intr_mask = 0;
   if (MI_register.w_mi_intr_mask_reg & 0x200) MI_register.PI_intr_mask = 1;
   if (MI_register.w_mi_intr_mask_reg & 0x400) MI_register.DP_intr_mask = 0;
   if (MI_register.w_mi_intr_mask_reg & 0x800) MI_register.DP_intr_mask = 1;
   MI_register.mi_intr_mask_reg = ((MI_register.SP_intr_mask) |
				     (MI_register.SI_intr_mask << 1) |
				     (MI_register.AI_intr_mask << 2) |
				     (MI_register.VI_intr_mask << 3) |
				     (MI_register.PI_intr_mask << 4) |
				     (MI_register.DP_intr_mask << 5)
				     );
}

void update_SP()
{
   if (sp_register.w_sp_status_reg & 0x1)
     sp_register.halt = 0;
   if (sp_register.w_sp_status_reg & 0x2)
     sp_register.halt = 1;
   if (sp_register.w_sp_status_reg & 0x4)
     sp_register.broke = 0;
   if (sp_register.w_sp_status_reg & 0x8)
     {
	MI_register.mi_intr_reg &= 0xFFFFFFFE;
	if (!dynacore && interpcore) check_interupt();
     }
   if (sp_register.w_sp_status_reg & 0x20)
     sp_register.single_step = 0;
   if (sp_register.w_sp_status_reg & 0x40)
     sp_register.single_step = 1;
   if (sp_register.w_sp_status_reg & 0x80)
     sp_register.intr_break = 0;
   if (sp_register.w_sp_status_reg & 0x100)
     sp_register.intr_break = 1;
   if (sp_register.w_sp_status_reg & 0x200)
     sp_register.signal0 = 0;
   if (sp_register.w_sp_status_reg & 0x400)
     sp_register.signal0 = 1;
   if (sp_register.w_sp_status_reg & 0x800)
     sp_register.signal1 = 0;
   if (sp_register.w_sp_status_reg & 0x2000)
     sp_register.signal2 = 0;
   if (sp_register.w_sp_status_reg & 0x4000)
     sp_register.signal2 = 1;
   if (sp_register.w_sp_status_reg & 0x8000)
     sp_register.signal3 = 0;
   if (sp_register.w_sp_status_reg & 0x20000)
     sp_register.signal4 = 0;
   if (sp_register.w_sp_status_reg & 0x80000)
     sp_register.signal5 = 0;
   if (sp_register.w_sp_status_reg & 0x200000)
     sp_register.signal6 = 0;
   if (sp_register.w_sp_status_reg & 0x800000)
     sp_register.signal7 = 0;
   if (sp_register.w_sp_status_reg & (~0xAAEFEF))
     {
	printf("unhandled write in sp status reg\n");
	printf("%x\n",(unsigned int)
	       (sp_register.w_sp_status_reg)&(~0xAAEFEF));
	stop=1;
     }
   sp_register.sp_status_reg = ((sp_register.halt) |
				  (sp_register.broke << 1) |
				  (sp_register.single_step << 5) |
				  (sp_register.intr_break << 6) |
				  (sp_register.signal0 << 7) |
				  (sp_register.signal1 << 8) |
				  (sp_register.signal2 << 9) |
				  (sp_register.signal3 << 10) |
				  (sp_register.signal4 << 11) |
				  (sp_register.signal5 << 12) |
				  (sp_register.signal6 << 13) |
				  (sp_register.signal7 << 14)
				  );
   if (!sp_register.halt && !sp_register.broke)
     {
	if (SP_DMEM[0xFC0/4] == 1)
	  {
	     processDList();
	     sp_register.sp_status_reg |= 0x203;
	     sp_register.signal2 = 1;
	     sp_register.broke = 1;
	     if (sp_register.intr_break)
	       {
		  MI_register.mi_intr_reg |= 0x1;
		  if (!dynacore && interpcore) check_interupt();
		  else gen_mi();
	       }
	     dpc_register.dpc_status &= ~2;
	     sp_register.halt = 1;
	  }
	else if (SP_DMEM[0xFC0/4] == 2)
	  {
	     sp_register.sp_status_reg |= 0x203;
	     sp_register.signal2 = 1;
	     sp_register.broke = 1;
	     if (sp_register.intr_break)
	       {
		  MI_register.mi_intr_reg |= 0x1;
		  if (!dynacore && interpcore) check_interupt();
		  else gen_mi();
	       }
	     sp_register.halt = 1;
	  }
	else
	  {
	     printf("other task\n");
	     sp_register.sp_status_reg |= 0x203;
	     sp_register.signal2 = 1;
	     sp_register.broke = 1;
	     if (sp_register.intr_break)
	       {
		  MI_register.mi_intr_reg |= 0x1;
		  gen_mi();
	       }
	     sp_register.halt = 1;
	  }
	//printf("unknown task type\n");
	/*if (hle) execute_dlist();
	//if (hle) processDList();
	else sp_register.halt = 0;*/
     }
}

void update_DPC()
{
   if (dpc_register.w_dpc_status & 0x1)
     dpc_register.xbus_dmem_dma = 0;
   if (dpc_register.w_dpc_status & 0x2)
     dpc_register.xbus_dmem_dma = 1;
   if (dpc_register.w_dpc_status & 0x4)
     dpc_register.freeze = 0;
   if (dpc_register.w_dpc_status & 0x8)
     dpc_register.freeze = 1;
   if (dpc_register.w_dpc_status & (~0x3CF))
     {
	printf("unhandled write in dpc status reg\n");
	printf("%x\n",(unsigned int)
	       (dpc_register.w_dpc_status)&(~0x3CF));
	stop=1;
     }
   dpc_register.dpc_status = ((dpc_register.xbus_dmem_dma) |
			      (dpc_register.freeze << 1)
			      );
}

void write_nothing()
{
}

void read_nothing()
{
   *rdword = 0;
}

void read_nothingh()
{
   *rdword = 0;
}

void read_nothingb()
{
   *rdword = 0;
}

void read_nomem()
{
#ifdef EMU64_DEBUG
   if ((address >= 0x80000000) && (address < 0xC0000000)) {
      printf("error : reading a word at %x PC=%x\n",
	     (unsigned int)address, (unsigned int)PC->addr);
      return;
   }
#endif
   address = virtual_to_physical_address(address,0);
   if (readmem[address>>16] == read_nomem) {
      printf("error : recursive read_nomem @ %x\n", (int)address);
      //stop = 1;
      return;
   }
   read_word_in_memory();
}

void read_nomemd()
{
#ifdef EMU64_DEBUG
   if ((address >= 0x80000000) && (address < 0xC0000000)) {
      printf("error : reading a double word at %x PC=%x\n",
	     (unsigned int)address, (unsigned int)PC->addr);
      return;
   }
#endif
   address = virtual_to_physical_address(address,0);
   if (readmemd[address>>16] == read_nomemd) {
      printf("error : recursive read_nomemd\n");
      stop = 1;
      return;
   }
   read_dword_in_memory();
}

void read_nomemb()
{
#ifdef EMU64_DEBUG
   if ((address >= 0x80000000) && (address < 0xC0000000)) {
      printf("error : reading a byte at %x PC=%x\n",
	     (unsigned int)address, (unsigned int)PC->addr);
   }
#endif
   address = virtual_to_physical_address(address,0);
   if (readmemb[address>>16] == read_nomemb) {
      printf("error : recursive read_nomemb\n");
      stop = 1;
      return;
   }
   read_byte_in_memory();
}

void read_nomemh()
{
#ifdef EMU64_DEBUG
   if ((address >= 0x80000000) && (address < 0xC0000000)) {
      printf("error : reading 2 bytes at %x PC=%x\n",
	     (unsigned int)address, (unsigned int)PC->addr);
   }
#endif
   address = virtual_to_physical_address(address,0);
   if (readmemh[address>>16] == read_nomemh) {
      printf("error : recursive read_nomemh\n");
      stop = 1;
      return;
   }
   read_hword_in_memory();
}

void write_nomem()
{
#ifdef EMU64_DEBUG
   if ((address >= 0x80000000) && (address < 0xC0000000)) {
      printf("error : writing a word at %x PC=%x\n",
	     (unsigned int)address, (unsigned int)PC->addr);
   }
#endif
   address = virtual_to_physical_address(address,1);
   if (writemem[address>>16] == write_nomem) {
      printf("error : recursive write_nomem\n");
      stop = 1;
      return;
   }
   write_word_in_memory();
}

void write_nomemd()
{
#ifdef EMU64_DEBUG
   if ((address >= 0x80000000) && (address < 0xC0000000)) {
      printf("error : writing double word at %x PC=%x\n",
	     (unsigned int)address, (unsigned int)PC->addr);
   }
#endif
   address = virtual_to_physical_address(address,1);
   if (writememd[address>>16] == write_nomemd) {
      printf("error : recursive write_nomemd\n");
      stop = 1;
      return;
   }
   write_dword_in_memory();
}

void write_nomemb()
{
#ifdef EMU64_DEBUG
   if ((address >= 0x80000000) && (address < 0xC0000000)) {
      printf("error : writing a byte at %x PC=%x\n",
	     (unsigned int)address, (unsigned int)PC->addr);
   }
#endif
   address = virtual_to_physical_address(address,1);
   if (writememb[address>>16] == write_nomemb) {
      printf("error : recursive write_nomemb\n");
      stop = 1;
      return;
   }
   write_byte_in_memory();
}

void write_nomemh()
{
#ifdef EMU64_DEBUG
   if ((address >= 0x80000000) && (address < 0xC0000000)) {
      printf("error : writing 2 bytes at %x PC=%x\n",
	     (unsigned int)address, (unsigned int)PC->addr);
   }
#endif
   address = virtual_to_physical_address(address,1);
   if (writememh[address>>16] == write_nomemh) {
      printf("error : recursive write_nomemh\n");
      stop = 1;
      return;
   }
   write_hword_in_memory();
}

void write_rdramd()
{
   *((unsigned long *)(rdramb + (address & 0xFFFFFF))) = dword >> 32;
   *((unsigned long *)(rdramb + (address & 0xFFFFFF) + 4 )) = dword & 0xFFFFFFFF;
}

void write_rdram()
{
   *((unsigned long *)(rdramb + (address & 0xFFFFFF))) = word;
}

void write_rdram_int()
{
   if (*address_low < 0x300)
     {
	int_handler.valid = 0;
	if (dynacore)
	  {
	     if (int_handler.code)
	       {
		  free(int_handler.code);
		  int_handler.code = NULL;
	       }
	  }
     }
   *((unsigned long *)(rdramb + (address & 0xFFFFFF))) = word;
}

void write_rdramh()
{
   *(unsigned short *)
     ((rdramb + ((address & 0xFFFFFF)^S16))) = hword;
}

void write_rdram_inth()
{
   if (*address_low < 0x300)
     {
	int_handler.valid = 0;
	if (dynacore)
	  {
	     if (int_handler.code)
	       {
		  free(int_handler.code);
		  int_handler.code = NULL;
	       }
	  }
     }
   *(unsigned short *)
     ((rdramb + ((address & 0xFFFFFF)^S16))) = hword;
}

void write_rdramb()
{
   *((rdramb + ((address & 0xFFFFFF)^S8))) = byte;
}

void write_rdram_intb()
{
   if (*address_low < 0x300)
     {
	int_handler.valid = 0;
	if (dynacore)
	  {
	     if (int_handler.code)
	       {
		  free(int_handler.code);
		  int_handler.code = NULL;
	       }
	  }
     }
   *((rdramb + ((address & 0xFFFFFF) ^S8))) = byte;
}

void write_rdram_intd()
{
   if (*address_low < 0x300)
     {
	int_handler.valid = 0;
	if (dynacore)
	  {
	     if (int_handler.code)
	       {
		  free(int_handler.code);
		  int_handler.code = NULL;
	       }
	  }
     }
   *((unsigned long *)(rdramb + (address & 0xFFFFFF))) = dword >> 32;
   *((unsigned long *)(rdramb + (address & 0xFFFFFF) + 4 )) = dword & 0xFFFFFFFF;
}

void read_rdram()
{;
   *rdword = *((unsigned long *)(rdramb + (address & 0xFFFFFF)));
}

void read_rdramd()
{
   *rdword = ((unsigned long long int)(*(unsigned long *)(rdramb + (address & 0xFFFFFF))) << 32) |
     ((*(unsigned long *)(rdramb + (address & 0xFFFFFF) + 4)));
}

void read_rdramb()
{
   *rdword = *(rdramb + ((address & 0xFFFFFF)^S8));
}

void read_rdramh()
{
   *rdword = *((unsigned short *)(rdramb + ((address & 0xFFFFFF)^S16)));
}

void write_actual_block()
{
   if (address > PC->addr)
     {
	int i;
	for (i=((address & 0x1FFFFFFF) >> 16);
	     i<((actual->end & 0x1FFFFFFF) >> 16); i++)
	  {
	     writemem[0x8000 + i] = write_rdram;
	     writemem[0xa000 + i] = write_rdram;
	     writememb[0x8000 + i] = write_rdramb;
	     writememb[0xa000 + i] = write_rdramb;
	     writememd[0x8000 + i] = write_rdramd;
	     writememd[0xa000 + i] = write_rdramd;
	     writememh[0x8000 + i] = write_rdramh;
	     writememh[0xa000 + i] = write_rdramh;
	  }
	actual->end = address - 4;
	actual->length = actual->end - actual->start;
	*((unsigned long *)(rdramb + (address & 0xFFFFFF))) = word;
	return;
     }
   if (address < actual->start)
     {
	if (address < 0x80000300) write_rdram_int();
	else write_rdram();
	return;
     }
   printf("write_actual_block()  l'addresse %x PC=%x\n",
	  (unsigned int)address, (unsigned int)PC->addr);
   stop=1;
}

void write_actual_blockb()
{
   printf("write_actual_blockb()  l'addresse %x PC=%x\n",
	  (unsigned int)address, (unsigned int)PC->addr);
   stop=1;
}

void write_actual_blockd()
{
   printf("write_actual_blockd()  l'addresse %x PC=%x\n",
	  (unsigned int)address, (unsigned int)PC->addr);
   stop=1;
}

void write_actual_blockh()
{
   if (address > PC->addr)
     {
	int i;
	for (i=((address & 0x1FFFFFFF) >> 16);
	     i<((actual->end & 0x1FFFFFFF) >> 16); i++)
	  {
	     writemem[0x8000 + i] = write_rdram;
	     writemem[0xa000 + i] = write_rdram;
	     writememb[0x8000 + i] = write_rdramb;
	     writememb[0xa000 + i] = write_rdramb;
	     writememd[0x8000 + i] = write_rdramd;
	     writememd[0xa000 + i] = write_rdramd;
	     writememh[0x8000 + i] = write_rdramh;
	     writememh[0xa000 + i] = write_rdramh;
	  }
	actual->end = address - 4;
	actual->length = actual->end - actual->start;
	*((rdramb + ((address & 0xFFFFFF)^S16))) = byte;
	return;
     }
   printf("write_actual_blockh()  l'addresse %x PC=%x\n",
	  (unsigned int)address, (unsigned int)PC->addr);
   stop=1;
}

void write_rdramreg()
{
#ifdef EMU64_DEBUG
   if (!readrdramreg[*address_low])
     {
	printf("error : writing a word at %x (rdramreg) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   *readrdramreg[*address_low] = word;
}

void read_rdramreg()
{
#ifdef EMU64_DEBUG
   if (!readrdramreg[*address_low])
     {
	printf("error : reading a word at %x (rdramreg) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	*rdword = 0;
	return;
     }
#endif
   *rdword = *(readrdramreg[*address_low]);
}

void read_rsp_mem()
{
   if (*address_low < 0x1000)
     {
	*rdword = *((unsigned long *)(SP_DMEMb + (*address_low)));
     }
   else if (*address_low < 0x2000)
     {
	*rdword = *((unsigned long *)(SP_IMEMb + (*address_low&0xFFF)));
	/*printf("aa:%x, %x, %x, %x\n", *address_low, address, 
	       *((unsigned long*)(SP_IMEMb + 0xf94)),
	       SP_IMEM[0xf94/4]);*/
     }
   else
     read_nomem();
}

void read_rsp_memb()
{
   if (*address_low < 0x1000)
     {
	*rdword = *(SP_DMEMb + (*address_low^S8));
     }
   else if (*address_low < 0x2000)
     {
	*rdword = *(SP_IMEMb + ((*address_low&0xFFF)^S8));
     }
   else
     read_nomemb();
}

void write_rsp_memd()
{
   if (*address_low < 0x1000)
     {
	*((unsigned long *)(SP_DMEMb + *address_low)) = dword >> 32;
	*((unsigned long *)(SP_DMEMb + *address_low + 4 )) = dword & 0xFFFFFFFF;
     }
   else if (*address_low < 0x2000)
     {
	printf("2\n");
     }
   else
     read_nomemd();
}

void write_rsp_mem()
{
   if (*address_low < 0x1000)
     {
	*((unsigned long *)(SP_DMEMb + (*address_low))) = word;
	sp.valid = 0;
	if (dynacore)
	  {
	     if (sp.code)
	       {
		  free(sp.code);
		  sp.code = NULL;
	       }
	  }
	return;
     }
   else
     {
	*((unsigned long *)(SP_IMEMb + (*address_low&0xFFF))) = word;
	return;
     }
}

void write_rsp_memb()
{
   if (*address_low < 0x1000)
     {
	*(SP_DMEMb + (*address_low^S8)) = byte;
	sp.valid = 0;
	if (dynacore)
	  {
	     if (sp.code)
	       {
		  free(sp.code);
		  sp.code = NULL;
	       }
	  }
	return;
     }
   else
     {
	*(SP_IMEMb + ((*address_low&0xFFF)^S8)) = byte;
	return;
     }
}

void read_rsp_reg()
{
#ifdef EMU64_DEBUG
   if (!readrspreg[*address_low])
     {
	printf("error : reading a word at %x (rspreg) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   *rdword = *(readrspreg[*address_low]);
}

void write_rsp_reg()
{
#ifdef EMU64_DEBUG
   if (!readrspreg[*address_low])
     {
	printf("error : writing a word at %x (rspreg) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   if (*address_low == 0x8)
     {
	sp_register.sp_rd_len_reg = word;
	dma_sp_write();
	return;
     }
   if (*address_low == 0x10)
     {
	sp_register.w_sp_status_reg = word;
	update_SP();
	return;
     }
   if (*address_low == 0x1C)
     {
	sp_register.sp_semaphore_reg = 0;
	return;
     }
   *readrspreg[*address_low] = word;
}

void read_rsp()
{
#ifdef EMU64_DEBUG
   if (!readrsp[*address_low])
     {
	printf("error : reading a word at %x (rsp) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	*rdword = 0;
	return;
     }
#endif
   *rdword = *(readrsp[*address_low]);
}

void write_rsp()
{
#ifdef EMU64_DEBUG
   if (!readrsp[*address_low])
     {
	printf("error : writing a word at %x (rsp) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   if (*address_low == 0x0)
     {
	RSP_PC_REG = word & 0xFFC;
	return;
     }
   *readrsp[*address_low] = word;
}

void read_dp()
{
#ifdef EMU64_DEBUG
   if (!readdp[*address_low])
     {
	printf("error : reading a word at %x (dp) PC=%x\n", 
	       (unsigned int)address, (unsigned int)PC->addr);
	*rdword = 0;
	return;
     }
#endif
   *rdword = *(readdp[*address_low]);
}

void write_dp()
{
#ifdef EMU64_DEBUG
   if (!readdp[*address_low])
     {
	printf("error : writing a word at %x (dp) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   switch(*address_low)
     {
      case 0x0:
	dpc_register.dpc_start = word;
	dpc_register.dpc_current = word;
	return;
      case 0x4:
	dpc_register.dpc_end =word;
	// toto : process RDP list !
	return;
      case 0xC:
	dpc_register.w_dpc_status = word;
	update_DPC();
	return;
     }
   printf("error in write_dp()\n");
}

void read_mi()
{
#ifdef EMU64_DEBUG
   if (!readmi[*address_low])
     {
	printf("error : reading a word at %x (mi) PC=%x\n", 
	       (unsigned int)address, (unsigned int)PC->addr);
	*rdword = 0;
	return;
     }
#endif
   *rdword = *(readmi[*address_low]);
}

void write_mi()
{
#ifdef EMU64_DEBUG
   if (!readmi[*address_low])
     {
	printf("error : writing a word at %x (mi) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   switch(*address_low)
     {
      case 0x0:
	MI_register.w_mi_init_mode_reg = word;
	update_MI_init_mode_reg();
	return;
      case 0xC:
	MI_register.w_mi_intr_mask_reg = word;
	update_MI_intr_mask_reg();
	if (!dynacore && interpcore) check_interupt();
	else gen_mi();
	return;
     }
   printf("error in write_mi()\n");
}

void write_vi()
{
#ifdef EMU64_DEBUG
   if (!readvi[*address_low])
     {
	printf("error : writing a word at %x (vi) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   switch(*address_low)
     {
      case 0x0:
	if (vi_register.vi_status != word)
	  {
	     vi_register.vi_status = word;
	     //update_config_display();
	     viStatusChanged();
	  }
	else vi_register.vi_status = word;
	return;
      case 0x4:
	vi_register.vi_origin = word;
	//update_config_display();
	return;
      case 0x8:
	if (vi_register.vi_width != word)
	  {
	     vi_register.vi_width = word;
	     //update_config_display();
	     viWidthChanged();
	  }
	else vi_register.vi_width = word;
	return;
      case 0x10:
	MI_register.mi_intr_reg &= 0xFFFFFFF7;
	if (!dynacore && interpcore) check_interupt();
	return;
     }
   *readvi[*address_low] = word;
}

void read_vi()
{
#ifdef EMU64_DEBUG
   if (!readvi[*address_low])
     {
	printf("error : reading a word at %x (vi) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	*rdword = 0;
	return;
     }
#endif
   if (*address_low == 0x10)
     {
#ifdef ORIGINAL
	*rdword = (((next_interupt - (Count+debug_count))/1220)*2);
	return;
#else
	if(vi_register.vi_current < 512) vi_register.vi_current +=2;
	else vi_register.vi_current = 0;
	*rdword = vi_register.vi_current;
	return;
#endif
     }
   *rdword = *(readvi[*address_low]);
}

void read_ai()
{
#ifdef EMU64_DEBUG
   if (!readai[*address_low])
     {
	printf("error : reading a word at %x (ai) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	*rdword = 0;
	return;
     }
#endif
   *rdword = *(readai[*address_low]);
}

void write_ai()
{
#ifdef EMU64_DEBUG
   if (!readai[*address_low])
     {
	printf("error : writing a word at %x (ai) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   if (*address_low == 0x4)
     {
	//*readai[*address_low] = word;
	MI_register.mi_intr_reg |= 0x04;
	check_interupt();
	//send_sound();
	return;
     }
   if (*address_low == 0xC)
     {
	MI_register.mi_intr_reg &= 0xFFFFFFFB;
	if (!dynacore && interpcore) check_interupt();
	return;
     }
   *readai[*address_low] = word;
}

void write_pi()
{
#ifdef EMU64_DEBUG
   if (!readpi[*address_low])
     {
	printf("error : writing a word at %x (pi) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   switch(*address_low)
     {
      case 0x08:
	pi_register.pi_rd_len_reg = word;
	dma_pi_read();
	return;
      case 0x0C:
	pi_register.pi_wr_len_reg = word;
	dma_pi_write();
	return;
      case 0x10:
	if (word) MI_register.mi_intr_reg &= 0xFFFFFFEF;
	if (!dynacore && interpcore) check_interupt();
	return;
      case 0x14:
      case 0x18:
      case 0x1c:
      case 0x20:
	*readpi[*address_low] = word & 0xFF;
     }
   *readpi[*address_low] = word;
}

void read_pi()
{
#ifdef EMU64_DEBUG
   if (!readpi[*address_low])
     {
	printf("error : reading a word at %x (pi) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	*rdword = 0;
	return;
     }
#endif
   *rdword = *(readpi[*address_low]);
}

void read_ri()
{
#ifdef EMU64_DEBUG
   if (!readri[*address_low])
     {
	printf("error : reading a word at %x (ri) PC=%x\n", 
	       (unsigned int)address, (unsigned int)PC->addr);
	*rdword = 0;
	return;
     }
#endif
   *rdword = *(readri[*address_low]);
}

void write_ri()
{
#ifdef EMU64_DEBUG
   if (!readri[*address_low])
     {
	printf("error : writing a word at %x (ri) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   if (*address_low == 0x8) word=0;
   *readri[*address_low] = word;
}

void read_si()
{
#ifdef EMU64_DEBUG
   if (!readsi[*address_low])
     {
	printf("error : reading a word at %x (si) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	*rdword = 0;
	return;
     }
#endif
   *rdword = *(readsi[*address_low]);
}

void write_si()
{
#ifdef EMU64_DEBUG
   if (!readsi[*address_low])
     {
	printf("error : writing a word at %x (si) PC=%x\n",
	       (unsigned int)address, (unsigned int)PC->addr);
	return;
     }
#endif
   switch(*address_low)
     {
      case 0x0:
	si_register.si_dram_addr = word;
	return;
      case 0x4:
	si_register.si_pif_addr_rd64b = word;
	dma_si_read();
	return;
      case 0x10:
	si_register.si_pif_addr_wr64b = word;
	dma_si_write();
	return;
      case 0x18:
	MI_register.mi_intr_reg &= 0xFFFFFFFD;
	if (!dynacore && interpcore) check_interupt();
	return;
     }
   printf("error in write_si()\n");
}

void read_rom()
{
   *rdword = *((unsigned long *)(rom + (address & 0x01FFFFFF)));
}

void read_pif()
{
#ifdef EMU64_DEBUG
   if ((*address_low > 0x7FF) || (*address_low < 0x7C0))
     {
	printf("error in reading a word in PIF\n");
	*rdword = 0;
	return;
     }
#endif
   //update_pif();
   *rdword = sl(*((unsigned long *)(PIF_RAMb + (address & 0x7FF) - 0x7C0)));
}

void read_pifb()
{
#ifdef EMU64_DEBUG
   if ((*address_low > 0x7FF) || (*address_low < 0x7C0))
     {
	printf("error in reading a byte in PIF\n");
	*rdword = 0;
	return;
     }
#endif
   *rdword = *(PIF_RAMb + ((address & 0x7FF) - 0x7C0));
}

void write_pif()
{
#ifdef EMU64_DEBUG
   if ((*address_low > 0x7FF) || (*address_low < 0x7C0))
     {
	printf("error in writing a word in PIF\n");
	return;
     }
#endif
   *((unsigned long *)(PIF_RAMb + (address & 0x7FF) - 0x7C0)) = sl(word);
   if ((address & 0x7FF) == 0x7FC)
     update_pif_write();
}

void write_flashram_command()
{
   if (use_flashram != -1 && *address_low == 0)
     {
	flashram_command(word);
	use_flashram = 1;
     }
   else
     printf("unknown write in write_flashram_command\n");
}

void read_flashram_status()
{
   if (use_flashram != -1 && *address_low == 0)
     {
	*rdword = flashram_status();
	use_flashram = 1;
     }
   else
     printf("unknown read in read_flashram_status\n");
}
